pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Dialogs
import QtQuick.Layouts 1.15
import QtQuick.Window 2.15
import QtCore 6.4 as QtCore

import styles 1.0
import features.settings 1.0
import features.welcome 1.0
import features.updates 1.0
import components.controls 1.0
import components.notifications 1.0
import features.cut 1.0
import features.transcription 1.0
import features.audiomix 1.0
import features.join 1.0
import features.export 1.0
import features.recovery 1.0
import features.feedback 1.0
import components.hints 1.0
import components.workspace 1.0
import "utils" as Utils

ApplicationWindow {
    id: window
    objectName: "appRootWindow"
    title: "Smart Media Cutter"
    color: Theme.background
    minimumWidth: 800
    minimumHeight: 600

    palette.window: Theme.background
    palette.windowText: Theme.textPrimary
    palette.base: Theme.isDark ? "#1e1e1e" : "#ffffff"
    palette.text: Theme.textPrimary
    palette.button: Theme.isDark ? "#3c3c3c" : "#e0e0e0"
    palette.buttonText: Theme.textPrimary
    palette.highlight: Theme.accent
    palette.highlightedText: "#ffffff"
    palette.placeholderText: Theme.isDark ? "#808080" : "#a0a0a0"
    palette.toolTipBase: Theme.isDark ? "#3c3c3c" : "#ffffdc"
    palette.toolTipText: Theme.textPrimary
    palette.link: Theme.isDark ? "#6ca4ff" : "#0066cc"
    palette.linkVisited: Theme.isDark ? "#b88aff" : "#8844cc"
    palette.alternateBase: Theme.isDark ? "#252525" : "#f5f5f5"
    palette.brightText: "#ffffff"
    palette.accent: Theme.accent
    palette.light: Theme.isDark ? "#505050" : "#f5f5f5"
    palette.midlight: Theme.isDark ? "#464646" : "#ebebeb"
    palette.dark: Theme.isDark ? "#282828" : "#b0b0b0"
    palette.mid: Theme.isDark ? "#323232" : "#c8c8c8"
    palette.shadow: Theme.isDark ? "#0a0a0a" : "#000000"

    // Required properties set by main_controller.py via createWithInitialProperties
    required property var appWorkspaceController
    required property var settingsModel
    required property var transcriptionController
    required property var themeController
    required property var updateController
    required property var languageController
    required property var feedbackService

    onWidthChanged: window._notifyHintLayoutChanged()
    onHeightChanged: window._notifyHintLayoutChanged()

    property int currentPageIndex: 0
    readonly property string currentWorkspaceKey: (window.tabModel && window.tabModel.length > window.currentPageIndex) ? window.tabModel[window.currentPageIndex].key : ""
    property var tabModel: [
        {
            key: "cut",
            label: qsTr("Cut"),
            icon: Theme.iconUrl("cut")
        },
        {
            key: "transcribe",
            label: qsTr("Transcribe"),
            icon: Theme.iconUrl("microphone")
        },
        {
            key: "audiomix",
            label: qsTr("Audio Mix"),
            icon: Theme.iconUrl("volume")
        },
        {
            key: "join",
            label: qsTr("Join"),
            icon: Theme.iconUrl("link"),
            premium: true
        }
    ]
    readonly property string projectExtension: window.appWorkspaceController.projectFileExtension

    QtObject {
        id: dialogFolders
        property url lastFolder: {
            const movies = QtCore.StandardPaths.standardLocations(QtCore.StandardPaths.MoviesLocation)
            if (movies.length) return movies[0]
            return QtCore.StandardPaths.writableLocation(QtCore.StandardPaths.HomeLocation)
        }
    }
    function formatDuration(seconds) {
        if (seconds <= 0) {
            return "";
        }
        const total = Math.floor(seconds);
        const hrs = Math.floor(total / 3600);
        const mins = Math.floor((total % 3600) / 60);
        const secs = total % 60;
        const secText = secs.toString().padStart(2, "0");
        if (hrs > 0) {
            const minText = mins.toString().padStart(2, "0");
            return `${hrs}:${minText}:${secText}`;
        }
        return `${mins}:${secText}`;
    }

    function _notifyHintLayoutChanged() {
        if (window.appWorkspaceController && window.appWorkspaceController.hintManager) {
            window.appWorkspaceController.hintManager.recalculate_positions()
        }
    }

    function _isProjectFile(path) {
        if (!path) return false;
        return path.toLowerCase().endsWith(window.projectExtension.toLowerCase());
    }

    function _openDroppedFile(path) {
        if (!path || !window.appWorkspaceController) return;
        if (window._isProjectFile(path)) {
            window.appWorkspaceController.loadProjectFrom(path);
        } else if (window.currentWorkspaceKey === "join" && window.appWorkspaceController.mediaStatus === "ready") {
            // In join workspace with media already loaded - add to join list
            window.appWorkspaceController.addMedia(path);
        } else {
            window.appWorkspaceController.openMedia(path);
        }
    }

    function _openMediaWithDialog() {
        loadMediaDialog.open();
        return true;
    }

    function _openProjectSaveDialog(existingPath) {
        saveProjectDialog.openForSave(existingPath || "");
        return true;
    }

    function _openProjectLoadDialog(folderPath) {
        openProjectDialog.openWithFolder(folderPath || "");
        return true;
    }

    header: Rectangle {
        height: 56
        color: Theme.surface

        Rectangle {
            anchors.bottom: parent.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            height: 1
            color: Theme.surfaceBorder
        }

        RowLayout {
            anchors.fill: parent
            anchors.leftMargin: Theme.pageMargin
            anchors.rightMargin: Theme.pageMargin
            spacing: Theme.spacingLg

            // Left: File Menu
            ToolButton {
                id: fileMenuButton
                objectName: "fileMenuButton"
                text: qsTr("File")
                Layout.preferredHeight: 36
                Layout.alignment: Qt.AlignVCenter
                onClicked: fileMenu.open()

                background: Rectangle {
                    color: fileMenuButton.down ? Theme.background : (fileMenuButton.hovered ? Theme.surfaceRaised : "transparent")
                    radius: 4
                }
                contentItem: Text {
                    text: fileMenuButton.text
                    font: fileMenuButton.font
                    color: Theme.textPrimary
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            // Update available indicator - shown when auto-check finds an update
            ToolButton {
                id: updateAvailableButton
                objectName: "updateAvailableButton"
                text: qsTr("New version!")
                Layout.preferredHeight: 36
                Layout.alignment: Qt.AlignVCenter

                // Track ignored version reactively for proper binding updates
                property string ignoredVersion: window.settingsModel ? window.settingsModel.ignored_update_version() : ""

                Connections {
                    target: window.settingsModel
                    function onIgnored_update_version_changed(version) {
                        updateAvailableButton.ignoredVersion = version
                    }
                }

                visible: window.updateController
                    ? (window.updateController.updateAvailable &&
                       window.updateController.latestVersion !== updateAvailableButton.ignoredVersion)
                    : false
                onClicked: updateAvailableDialog.open()

                background: Rectangle {
                    color: updateAvailableButton.down ? Theme.background : (updateAvailableButton.hovered ? Theme.surfaceRaised : "transparent")
                    radius: 4
                }
                contentItem: Text {
                    text: updateAvailableButton.text
                    font: updateAvailableButton.font
                    color: Theme.accent
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            // Undo History dropdown
            UndoHistoryButton {
                id: undoHistoryButton
                Layout.preferredHeight: 36
                Layout.alignment: Qt.AlignVCenter
                workspaceController: window.appWorkspaceController
                visible: window.appWorkspaceController ? window.appWorkspaceController.mediaStatus === "ready" : false
            }

            // Center: Status Pill
            Rectangle {
                Layout.fillWidth: true
                Layout.preferredHeight: 32
                Layout.alignment: Qt.AlignVCenter
                color: Theme.surfaceRaised
                radius: 16
                border.color: Theme.surfaceBorder
                border.width: 1

                Label {
                    id: mediaStatusLabel
                    anchors.centerIn: parent
                    width: parent.width - 32
                    horizontalAlignment: Text.AlignHCenter
                    elide: Text.ElideRight
                    color: Theme.textSecondary
                    text: window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "ready"
                          ? qsTr("Loaded: %1 (%2)")
                              .arg(window.appWorkspaceController.mediaDisplayName || qsTr("Unknown media"))
                              .arg(window.formatDuration(window.appWorkspaceController.mediaDuration))
                          : window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "loading"
                                ? qsTr("Loading media…")
                                : window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "error"
                                      ? window.appWorkspaceController.mediaError || qsTr("Failed to open media")
                                      : qsTr("No media loaded")
                }
            }

            // Right: Actions Group
            RowLayout {
                Layout.alignment: Qt.AlignVCenter
                spacing: Theme.spacingSm

                PrimaryButton {
                    id: exportButton
                    objectName: "exportButton"
                    Layout.preferredHeight: 36
                    text: qsTr("Export")
                    opacity: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready" ? 1.0 : 0.0) : 0.0
                    enabled: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready") : false
                    onClicked: {
                        if (window.appWorkspaceController && window.appWorkspaceController.hintManager) {
                            window.appWorkspaceController.hintManager.dismissHint("export_button")
                        }
                        exportOverlay.open()
                    }
                }

                Rectangle {
                    Layout.preferredWidth: 1
                    Layout.preferredHeight: 24
                    color: Theme.surfaceBorder
                    Layout.leftMargin: 4
                    Layout.rightMargin: 4
                }

                IconButton {
                    id: viewOptionsButton
                    objectName: "viewOptionsButton"
                    Layout.preferredWidth: 36
                    Layout.preferredHeight: 36
                    iconSource: Theme.iconUrl("eye")
                    opacity: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready" ? 1.0 : 0.0) : 0.0
                    enabled: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready") : false
                    onClicked: {
                        viewMenu.open()
                        if (window.appWorkspaceController && window.appWorkspaceController.hintManager) {
                            window.appWorkspaceController.hintManager.dismissHint("view_customization")
                        }
                    }

                    ToolTip.visible: hovered
                    ToolTip.delay: 200
                    ToolTip.text: qsTr("View options - Enable and Disable UI Views")
                }

                ThemeToggleSwitch {
                    Layout.preferredWidth: 88
                    Layout.preferredHeight: 36
                }

                IconButton {
                    id: settingsButton
                    objectName: "settingsButton"
                    Layout.preferredWidth: 36
                    Layout.preferredHeight: 36
                    iconSource: Theme.iconUrl("settings")
                    checkable: true
                    checked: settingsOverlay.visible
                    onClicked: settingsOverlay.visible ? settingsOverlay.close() : settingsOverlay.open()
                }
            }
        }

        Menu {
            id: fileMenu
            parent: fileMenuButton
            y: fileMenuButton.height

            MenuItem {
                text: qsTr("Open…")
                onTriggered: window._openMediaWithDialog()
            }

            MenuSeparator {
                contentItem: Rectangle {
                    implicitHeight: 1
                    color: Theme.surfaceBorder
                }
            }

            MenuItem {
                text: qsTr("Open project…")
                onTriggered: {
                    if (!window.appWorkspaceController) {
                        return
                    }
                    window._openProjectLoadDialog(Utils.PathUtils.urlToPath(saveProjectDialog.currentFolder))
                }
            }

            MenuItem {
                text: qsTr("Save project…")
                enabled: window.appWorkspaceController ? window.appWorkspaceController.mediaStatus === "ready" : false
                onTriggered: {
                    if (!window.appWorkspaceController) {
                        return
                    }
                    if (window.appWorkspaceController.projectPath && window.appWorkspaceController.saveProject()) {
                        return
                    }
                    var existing = window.appWorkspaceController.projectPath
                    window._openProjectSaveDialog(existing)
                }
            }

            MenuItem {
                text: qsTr("Save project as…")
                enabled: window.appWorkspaceController ? window.appWorkspaceController.mediaStatus === "ready" : false
                onTriggered: {
                    if (!window.appWorkspaceController) {
                        return
                    }
                    // Always open dialog, suggesting current path if exists
                    window._openProjectSaveDialog(window.appWorkspaceController.projectPath || "")
                }
            }

            Menu {
                id: recentProjectsMenu
                title: qsTr("Recent Projects")

                // Property to hold the list - updated when signal fires
                property var recentList: window.settingsModel ? window.settingsModel.recent_projects() : []
                property real calculatedWidth: 200

                onRecentListChanged: {
                    let maxWidth = 200;
                    for (let i = 0; i < recentList.length; i++) {
                        recentTextMetrics.text = Utils.PathUtils.formatRecentPath(recentList[i]);
                        maxWidth = Math.max(maxWidth, recentTextMetrics.width);
                    }
                    calculatedWidth = maxWidth + 48;
                }

                // Calculate width from longest entry
                TextMetrics {
                    id: recentTextMetrics
                }

                implicitWidth: calculatedWidth

                // Use enabled instead of visible - binding visible on a submenu
                // during construction causes segfaults in Qt 6.8+ (PySide6 6.10+)
                enabled: recentList.length > 0

                Instantiator {
                    model: recentProjectsMenu.recentList
                    delegate: MenuItem {
                        required property string modelData
                        required property int index
                        text: Utils.PathUtils.formatRecentPath(modelData)

                        ToolTip.visible: hovered
                        ToolTip.delay: 500
                        ToolTip.text: modelData

                        onTriggered: {
                            recentProjectsMenu.close()
                            fileMenu.close()
                            if (!window.appWorkspaceController) {
                                return
                            }
                            // Check if project is dirty - only show dialog if there are unsaved changes
                            if (window.appWorkspaceController.projectDirty) {
                                unsavedChangesDialog.pendingPath = modelData
                                unsavedChangesDialog.open()
                            } else {
                                window.appWorkspaceController.loadProjectFrom(modelData)
                            }
                        }
                    }
                    onObjectAdded: (index, object) => recentProjectsMenu.insertItem(index, object)
                    onObjectRemoved: (index, object) => recentProjectsMenu.removeItem(object)
                }

                Connections {
                    target: window.settingsModel || null
                    function onRecent_projects_changed() {
                        // Update the property - this forces Instantiator to refresh
                        recentProjectsMenu.recentList = window.settingsModel.recent_projects()
                    }
                }
            }

            MenuSeparator {
                contentItem: Rectangle {
                    implicitHeight: 1
                    color: Theme.surfaceBorder
                }
            }

            MenuItem {
                text: qsTr("Quit")
                onTriggered: window.close()
            }
        }

        Menu {
            id: viewMenu
            parent: viewOptionsButton
            y: viewOptionsButton.height

            property string wk: window.currentWorkspaceKey

            onAboutToShow: {
                wk = window.currentWorkspaceKey
            }

            // Cut workspace options
            MenuItem {
                visible: viewMenu.wk === "cut"
                height: visible ? implicitHeight : 0
                checkable: true
                text: qsTr("Left tools panel")
                checked: {
                    var dflt = ViewDefaults.getDefault("cut", "showToolsSidebar")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view("cut")
                    return cfg.hasOwnProperty("showToolsSidebar") ? !!cfg["showToolsSidebar"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag("cut", "showToolsSidebar", checked)
            }

            MenuItem {
                visible: viewMenu.wk === "cut"
                height: visible ? implicitHeight : 0
                checkable: true
                text: qsTr("Preview frame strip")
                checked: {
                    var dflt = ViewDefaults.getDefault("cut", "showFrameStrip")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view("cut")
                    return cfg.hasOwnProperty("showFrameStrip") ? !!cfg["showFrameStrip"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag("cut", "showFrameStrip", checked)
            }

            MenuItem {
                visible: viewMenu.wk === "cut"
                height: visible ? implicitHeight : 0
                checkable: true
                text: qsTr("Timecode display")
                checked: {
                    var dflt = ViewDefaults.getDefault("cut", "showPlayheadInfo")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view("cut")
                    return cfg.hasOwnProperty("showPlayheadInfo") ? !!cfg["showPlayheadInfo"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag("cut", "showPlayheadInfo", checked)
            }

            MenuItem {
                visible: viewMenu.wk === "cut"
                height: visible ? implicitHeight : 0
                checkable: true
                text: qsTr("Cut actions toolbar")
                checked: {
                    var dflt = ViewDefaults.getDefault("cut", "showCutActionsToolbar")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view("cut")
                    return cfg.hasOwnProperty("showCutActionsToolbar") ? !!cfg["showCutActionsToolbar"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag("cut", "showCutActionsToolbar", checked)
            }

            MenuSeparator {
                visible: viewMenu.wk === "cut"
                height: visible ? implicitHeight : 0
            }

            MenuItem {
                id: overviewTimelineItem
                visible: viewMenu.wk === "cut" || viewMenu.wk === "transcribe" || viewMenu.wk === "audiomix"
                checkable: true
                text: qsTr("Overview timeline")
                checked: {
                    var dflt = ViewDefaults.getDefault(viewMenu.wk, "showOverviewTimeline")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view(viewMenu.wk)
                    return cfg.hasOwnProperty("showOverviewTimeline") ? !!cfg["showOverviewTimeline"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag(viewMenu.wk, "showOverviewTimeline", checked)
            }

            MenuItem {
                id: audioTimelineItem
                visible: viewMenu.wk === "cut" || viewMenu.wk === "transcribe" || viewMenu.wk === "audiomix"
                checkable: true
                text: qsTr("Audio timeline")
                checked: {
                    var dflt = ViewDefaults.getDefault(viewMenu.wk, "showAudioTimeline")
                    if (!window.settingsModel) return dflt
                    var cfg = window.settingsModel.workspace_view(viewMenu.wk)
                    return cfg.hasOwnProperty("showAudioTimeline") ? !!cfg["showAudioTimeline"] : dflt
                }
                onToggled: window.settingsModel && window.settingsModel.set_workspace_view_flag(viewMenu.wk, "showAudioTimeline", checked)
            }

        }
    }

    ColumnLayout {
        id: contentLayout
        visible: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready") : false
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: parent.top
        anchors.bottom: mainTabs.top
        anchors.leftMargin: Theme.pageMargin
        anchors.rightMargin: Theme.pageMargin
        anchors.topMargin: Theme.pageMargin
        anchors.bottomMargin: Theme.spacingSm
        spacing: Theme.spacingSm

        StackLayout {
            id: mainStack
            objectName: "mainStack"
            Layout.fillWidth: true
            Layout.fillHeight: true
            currentIndex: window.currentPageIndex

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true

                CutWorkspaceTab {
                    id: workspaceTab
                    anchors.fill: parent
                    appWorkspaceController: window.appWorkspaceController
                    settingsModel: window.settingsModel
                }
            }

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true

                Flickable {
                    anchors.fill: parent
                    contentWidth: width
                    contentHeight: height
                    clip: true

                    Component {
                        id: transcriptionTabComponent
                        TranscriptionWorkspaceTab {
                            id: transcriptionTab
                            objectName: "transcriptionTab"
                            anchors.fill: parent
                            controller: window.transcriptionController
                            appController: window.appWorkspaceController
                            timelineBackend: window.appWorkspaceController.timelineBackend
                            settingsModel: window.settingsModel
                            languageController: window.languageController
                            toastManager: toastManager
                            onFeedbackRequested: feedbackOverlay.open()
                        }
                    }

                    Loader {
                        id: transcriptionLoader
                        anchors.fill: parent
                        active: !!window.transcriptionController
                        sourceComponent: transcriptionTabComponent
                    }
                }
            }

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true

                AudioMixWorkspaceTab {
                    id: audioMixTab
                    objectName: "audioMixTab"
                    anchors.fill: parent
                    appController: window.appWorkspaceController
                    timelineBackend: window.appWorkspaceController.timelineBackend
                    settingsModel: window.settingsModel
                }
            }

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true

                JoinWorkspaceTab {
                    id: joinTab
                    objectName: "joinTab"
                    anchors.fill: parent
                    appWorkspaceController: window.appWorkspaceController
                    settingsModel: window.settingsModel
                }
            }

        }

    }

    BottomTabBar {
        id: mainTabs
        objectName: "mainTabBar"
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.bottom: parent.bottom
        visible: window.appWorkspaceController ? (window.appWorkspaceController.mediaStatus === "ready") : false
        model: window.tabModel
        currentIndex: window.currentPageIndex
        caption: qsTr("Smart Media Cutter %1").arg(window.appWorkspaceController ? window.appWorkspaceController.appVersion : "")
        licenseManager: window.appWorkspaceController ? window.appWorkspaceController.licenseManager : null
        onTabRequested: function(tabIndex) {
            // Check if this is a premium tab and license is invalid
            var tabEntry = window.tabModel[tabIndex]
            if (tabEntry && tabEntry.premium) {
                var lm = window.appWorkspaceController ? window.appWorkspaceController.licenseManager : null
                if (!lm || !lm.isValid()) {
                    joinPremiumPopup.open()
                    return
                }
            }
            window.currentPageIndex = tabIndex
            // Update hint manager with current workspace
            if (window.appWorkspaceController && window.appWorkspaceController.hintManager && tabEntry) {
                window.appWorkspaceController.hintManager.set_current_workspace(tabEntry.key)
            }
        }
    }

    Item {
        id: emptyStateOverlay
        anchors.left: contentLayout.left
        anchors.right: contentLayout.right
        anchors.top: contentLayout.top
        anchors.bottom: contentLayout.bottom
        z: 999
        visible: !window.appWorkspaceController || (window.appWorkspaceController.mediaStatus !== "ready" && window.appWorkspaceController.mediaStatus !== "loading")
        property bool dragging: false

        Rectangle {
            anchors.fill: parent
            color: Theme.background
        }

        DropArea {
            anchors.fill: parent
            onEntered: function(drag) {
                emptyStateOverlay.dragging = true
            }
            onExited: function() {
                emptyStateOverlay.dragging = false
            }
            onDropped: function(drop) {
                if (drop.hasUrls && drop.urls.length) {
                    var path = Utils.PathUtils.urlToPath(drop.urls[0])
                    window._openDroppedFile(path)
                }
                emptyStateOverlay.dragging = false
            }
        }

        MouseArea {
            anchors.fill: parent
            cursorShape: Qt.PointingHandCursor
            onClicked: window._openMediaWithDialog()
        }

        Canvas {
            id: borderCanvas
            anchors.fill: parent
            onPaint: {
                var ctx = getContext("2d")
                ctx.reset()
                ctx.lineWidth = 2
                ctx.setLineDash([6, 6])
                ctx.strokeStyle = emptyStateOverlay.dragging ? Theme.accent : Theme.surfaceBorder
                var m = Theme.pageMargin * 2
                ctx.strokeRect(m, m, width - 2*m, height - 2*m)
            }
        }

        Connections {
            target: emptyStateOverlay
            function onDraggingChanged() {
                borderCanvas.requestPaint()
            }
        }

        Column {
            anchors.centerIn: parent
            spacing: Theme.spacingMd

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: qsTr("Drop media or project file here to start")
                font.pixelSize: 24
                font.weight: Font.Bold
                color: Theme.textPrimary
            }

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: qsTr("or click to browse")
                font.pixelSize: 14
                color: Theme.textSecondary
            }

            Button {
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load sneezing goat (dev)"
                onClicked: {
                    if (window.appWorkspaceController) {
                        window.appWorkspaceController.openMedia("test_input/sneezing_goat.mp4")
                    }
                }
            }

            Button {
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load tears of steel (dev)"
                onClicked: {
                    if (window.appWorkspaceController) {
                        window.appWorkspaceController.openMedia("/home/santtu/Videos/tears_of_steel.mp4")
                    }
                }
            }

            Button {
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load ai_short (dev)"
                onClicked: {
                    if (window.appWorkspaceController) {
                        window.appWorkspaceController.openMedia("/home/santtu/Videos//ai_short.mkv")
                    }
                }
            }

            Button {
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load LGCWEP687 (dev)"
                onClicked: {
                    if (window.appWorkspaceController) {
                        window.appWorkspaceController.openMedia("test_input/LGCWEP687.mp3")
                    }
                }
            }

            Button {
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load JFK speech (dev)"
                onClicked: {
                    if (window.appWorkspaceController) {
                        window.appWorkspaceController.openMedia("pywhispercpp/whisper.cpp/samples/jfk.mp3")
                    }
                }
            }

            Button {
                id: loadGoatPartsButton
                visible: window.appWorkspaceController ? window.appWorkspaceController.isDevMode : false
                anchors.horizontalCenter: parent.horizontalCenter
                text: "Load goat parts 1+2 (dev)"
                property bool pendingSecondFile: false
                onClicked: {
                    if (window.appWorkspaceController) {
                        pendingSecondFile = true
                        window.appWorkspaceController.openMedia("test_input/sneezing_goat_part1.mp4")
                    }
                }
                Connections {
                    target: window.appWorkspaceController
                    enabled: loadGoatPartsButton.pendingSecondFile
                    function onMediaStatusChanged() {
                        if (window.appWorkspaceController.mediaStatus === "ready" && loadGoatPartsButton.pendingSecondFile) {
                            loadGoatPartsButton.pendingSecondFile = false
                            window.appWorkspaceController.addMedia("test_input/sneezing_goat_part2.mp4")
                            window.currentPageIndex = 3
                        }
                    }
                }
            }
        }
    }

    Item {
        id: loadingMediaOverlay
        anchors.left: contentLayout.left
        anchors.right: contentLayout.right
        anchors.top: contentLayout.top
        anchors.bottom: contentLayout.bottom
        z: 999
        visible: !!(window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "loading")

        Rectangle {
            anchors.fill: parent
            color: Theme.background
        }

        Canvas {
            id: loadingBorderCanvas
            anchors.fill: parent
            onPaint: {
                var ctx = getContext("2d")
                ctx.reset()
                ctx.lineWidth = 2
                ctx.setLineDash([6, 6])
                ctx.strokeStyle = Theme.accent
                var m = Theme.pageMargin * 2
                ctx.strokeRect(m, m, width - 2*m, height - 2*m)
            }
        }

        Column {
            anchors.centerIn: parent
            spacing: Theme.spacingMd

            // loadingProgress < 0 means no audio loading yet (MediaContainer creation phase)
            property real progress: window.appWorkspaceController ? window.appWorkspaceController.loadingProgress : -1
            property bool hasAudioProgress: progress >= 0

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: qsTr("Opening media file for smooth editing")
                font.pixelSize: 24
                font.weight: Font.Bold
                color: Theme.textPrimary
            }

            BusyIndicator {
                anchors.horizontalCenter: parent.horizontalCenter
                running: loadingMediaOverlay.visible && !parent.hasAudioProgress
                visible: !parent.hasAudioProgress
            }

            ProgressBar {
                id: loadingProgressBar
                anchors.horizontalCenter: parent.horizontalCenter
                width: 300
                from: 0
                to: 1
                value: parent.progress
                visible: parent.hasAudioProgress
            }

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: parent.hasAudioProgress
                    ? qsTr("Loading audio: %1%").arg(Math.round(parent.progress * 100))
                    : qsTr("Please wait")
                font.pixelSize: 14
                color: Theme.textSecondary
            }
        }
    }

    Item {
        id: workspaceDragOverlay
        anchors.left: contentLayout.left
        anchors.right: contentLayout.right
        anchors.top: contentLayout.top
        anchors.bottom: contentLayout.bottom
        z: 1000
        // Always visible when media is ready, so DropArea can receive events
        visible: !!(window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "ready")
        property bool dragging: false

        // DropArea is always active to receive drag events
        DropArea {
            anchors.fill: parent
            enabled: true
            onEntered: function(drag) {
                if (drag.hasUrls && drag.urls.length) {
                    workspaceDragOverlay.dragging = true
                }
            }
            onExited: function() {
                workspaceDragOverlay.dragging = false
            }
            onDropped: function(drop) {
                workspaceDragOverlay.dragging = false
                if (drop.hasUrls && drop.urls.length) {
                    var path = Utils.PathUtils.urlToPath(drop.urls[0])
                    if (path && window.appWorkspaceController) {
                        // In join workspace with media, adding files doesn't need confirmation
                        var isJoinAdd = window.currentWorkspaceKey === "join" &&
                                        window.appWorkspaceController.mediaStatus === "ready" &&
                                        !window._isProjectFile(path)
                        // Check if project is dirty - only show dialog if there are unsaved changes
                        // and we're not just adding a file in join workspace
                        if (window.appWorkspaceController.projectDirty && !isJoinAdd) {
                            unsavedChangesDialog.pendingPath = path
                            unsavedChangesDialog.open()
                        } else {
                            window._openDroppedFile(path)
                        }
                    }
                }
            }
        }

        // Visual feedback only shown when dragging
        Rectangle {
            anchors.fill: parent
            color: Theme.background
            opacity: 0.95
            visible: workspaceDragOverlay.dragging
        }

        Canvas {
            id: workspaceBorderCanvas
            anchors.fill: parent
            visible: workspaceDragOverlay.dragging
            onPaint: {
                var ctx = getContext("2d")
                ctx.reset()
                ctx.lineWidth = 3
                ctx.setLineDash([8, 8])
                ctx.strokeStyle = Theme.accent
                var m = Theme.pageMargin * 2
                ctx.strokeRect(m, m, width - 2*m, height - 2*m)
            }
        }

        Connections {
            target: workspaceDragOverlay
            function onDraggingChanged() {
                workspaceBorderCanvas.requestPaint()
            }
        }

        Column {
            anchors.centerIn: parent
            spacing: Theme.spacingMd
            visible: workspaceDragOverlay.dragging

            property bool isJoinWorkspace: window.currentWorkspaceKey === "join"

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: parent.isJoinWorkspace ? qsTr("Add file to join list") : qsTr("Load new file?")
                font.pixelSize: 24
                font.weight: Font.Bold
                color: Theme.textPrimary
            }

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                text: parent.isJoinWorkspace ? qsTr("Drop to add this file to the project") : qsTr("This will replace the current project")
                font.pixelSize: 14
                color: Theme.textSecondary
            }
        }
    }

    SettingsOverlay {
        id: settingsOverlay
        onVisibleChanged: settingsButton.checked = visible
        parent: window.contentItem
        settingsModel: window.settingsModel
        themeController: window.themeController
        appWorkspaceController: window.appWorkspaceController
        updateController: window.updateController
        languageController: window.languageController
        onOpenFeedback: feedbackOverlay.open()
    }

    FeedbackOverlay {
        id: feedbackOverlay
        parent: window.contentItem
        feedbackService: window.feedbackService
        licenseManager: window.appWorkspaceController.licenseManager
    }

    Connections {
        target: window.feedbackService
        function onFeedbackSent() {
            toastManager.success(qsTr("Feedback sent"), qsTr("Thank you for your feedback!"))
        }
    }

    FileDialog {
        id: loadMediaDialog
        objectName: "loadMediaDialog"
        title: qsTr("Open File")
        nameFilters: [
            qsTr("Supported Files") + " (*.mp4 *.mov *.mkv *.mp3 *.wav *.flac *.aac *.smcproj)",
            qsTr("Media Files") + " (*.mp4 *.mov *.mkv *.mp3 *.wav *.flac *.aac)",
            qsTr("Project Files") + " (*.smcproj)",
            qsTr("All Files") + " (*)"
        ]
        fileMode: FileDialog.OpenFile
        currentFolder: dialogFolders.lastFolder

        onAccepted: {
            const filePath = Utils.PathUtils.urlToPath(selectedFile)
            if (!filePath) {
                return
            }
            dialogFolders.lastFolder = currentFolder

            // Check if project is dirty - only show dialog if there are unsaved changes
            if (window.appWorkspaceController && window.appWorkspaceController.projectDirty) {
                unsavedChangesDialog.pendingPath = filePath
                unsavedChangesDialog.open()
            } else {
                window._openDroppedFile(filePath)
            }
        }
    }

    FileDialog {
        id: saveProjectDialog
        objectName: "saveProjectDialog"
        title: qsTr("Save Project")
        nameFilters: [
            qsTr("Smart Media Cutter Project") + " (*" + window.projectExtension + ")",
            qsTr("All Files") + " (*)"
        ]
        fileMode: FileDialog.SaveFile
        defaultSuffix: window.projectExtension.length ? window.projectExtension.slice(1) : ""
        currentFolder: dialogFolders.lastFolder

        onAccepted: {
            let filePath = Utils.PathUtils.urlToPath(selectedFile)
            if (!filePath) {
                return
            }
            const normalizedPath = saveProjectDialog._ensureProjectExtension(filePath)
            dialogFolders.lastFolder = currentFolder
            if (window.appWorkspaceController) {
                window.appWorkspaceController.saveProjectTo(normalizedPath)
            }
        }

        function openForSave(existingPath) {
            let path = ""
            if (existingPath && existingPath.length) {
                path = saveProjectDialog._ensureProjectExtension(existingPath)
            } else if (window.appWorkspaceController) {
                // suggestedProjectFileName returns full absolute path
                path = window.appWorkspaceController.suggestedProjectFileName()
            }
            if (path) {
                currentFolder = Utils.PathUtils.pathToUrl(Utils.PathUtils.dirname(path))
                selectedFile = Utils.PathUtils.pathToUrl(path)
            }
            open()
        }

        function _ensureProjectExtension(path) {
            const normalized = Utils.PathUtils.normalizePath(path)
            const extension = window.projectExtension
            if (!normalized.length) {
                return ""
            }
            if (!extension.length) {
                return normalized
            }
            const lowerExt = extension.toLowerCase()
            if (normalized.toLowerCase().endsWith(lowerExt)) {
                return normalized
            }
            const slashIndex = normalized.lastIndexOf("/")
            const dotIndex = normalized.lastIndexOf(".")
            if (dotIndex > slashIndex) {
                return normalized.slice(0, dotIndex) + extension
            }
            return normalized + extension
        }
    }

    FileDialog {
        id: openProjectDialog
        objectName: "openProjectDialog"
        title: qsTr("Open Project")
        nameFilters: [
            qsTr("Smart Media Cutter Project") + " (*" + window.projectExtension + ")",
            qsTr("All Files") + " (*)"
        ]
        fileMode: FileDialog.OpenFile
        currentFolder: dialogFolders.lastFolder

        onAccepted: {
            let filePath = Utils.PathUtils.urlToPath(selectedFile)
            if (!filePath) {
                return
            }
            filePath = saveProjectDialog._ensureProjectExtension(filePath)
            dialogFolders.lastFolder = currentFolder

            // Check if project is dirty - only show dialog if there are unsaved changes
            if (window.appWorkspaceController && window.appWorkspaceController.projectDirty) {
                unsavedChangesDialog.pendingPath = filePath
                unsavedChangesDialog.open()
            } else if (window.appWorkspaceController) {
                window.appWorkspaceController.loadProjectFrom(filePath)
            }
        }

        function openWithFolder(folderPath) {
            if (folderPath && folderPath.length) {
                currentFolder = Utils.PathUtils.pathToUrl(Utils.PathUtils.normalizePath(folderPath))
            }
            open()
        }
    }

    ExportOverlay {
        id: exportOverlay
        parent: window.contentItem
        appWorkspaceController: window.appWorkspaceController
        settingsModel: window.settingsModel
        onRequestNavigateToAudioMix: mainTabs.tabRequested(2)  // Audio Mix tab via proper channels
    }

    // Hint tooltip for export overlay (needs separate instance above modal)
    HintTooltip {
        id: exportHintTooltip
        parent: Overlay.overlay
        anchors.fill: parent
        z: 9999
        hint: window.appWorkspaceController ? window.appWorkspaceController.hintManager.currentHint : null
        searchRoot: exportOverlay.contentItem
        hintManager: window.appWorkspaceController ? window.appWorkspaceController.hintManager : null
        // Must check targetItem to avoid flash at (0,0) before target is found
        visible: exportOverlay.visible && !!hint && !!targetItem && hint.targetObjectName && (
            hint.targetObjectName === "exportOutputPathField" ||
            hint.targetObjectName === "startExportButton" ||
            hint.targetObjectName === "timelineExportTab"
        )
    }

    // Recovery popup for corrupt MP4 files
    RecoveryPopup {
        id: recoveryPopup
        parent: Overlay.overlay
        anchors.centerIn: parent
        recoveryController: window.appWorkspaceController ? window.appWorkspaceController.recoveryController : null
        licenseManager: window.appWorkspaceController ? window.appWorkspaceController.licenseManager : null
        languageController: window.languageController
        toastManager: toastManager
        appWorkspaceController: window.appWorkspaceController

        onLoadRecoveredFile: function(path) {
            if (window.appWorkspaceController) {
                window.appWorkspaceController.openMedia(path)
            }
        }
    }

    // Handle corrupt MP4 detection
    Connections {
        target: window.appWorkspaceController
        function onCorrupt_mp4_detected(path) {
            recoveryPopup.open()
        }
    }

    Dialog {
        id: unsavedChangesDialog
        objectName: "unsavedChangesDialog"
        parent: window.contentItem
        anchors.centerIn: parent
        modal: true
        Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
        title: qsTr("Unsaved Changes")
        standardButtons: Dialog.NoButton

        property string pendingPath: ""

        ColumnLayout {
            spacing: Theme.spacingMd

            Label {
                Layout.fillWidth: true
                text: qsTr("You have unsaved changes. Would you like to save your project before continuing?")
                wrapMode: Text.WordWrap
            }

            RowLayout {
                Layout.alignment: Qt.AlignRight
                spacing: Theme.spacingSm

                SecondaryButton {
                    text: qsTr("Cancel")
                    onClicked: unsavedChangesDialog.reject()
                }

                SecondaryButton {
                    text: qsTr("Discard")
                    onClicked: {
                        // Discard changes and proceed
                        if (unsavedChangesDialog.pendingPath) {
                            window._openDroppedFile(unsavedChangesDialog.pendingPath)
                            unsavedChangesDialog.pendingPath = ""
                        }
                        unsavedChangesDialog.close()
                    }
                }

                PrimaryButton {
                    text: qsTr("Save")
                    onClicked: {
                        // Save project first, then proceed with the pending action
                        if (window.appWorkspaceController) {
                            if (window.appWorkspaceController.projectPath) {
                                // Has existing path - save directly
                                window.appWorkspaceController.saveProject()
                                if (unsavedChangesDialog.pendingPath) {
                                    window._openDroppedFile(unsavedChangesDialog.pendingPath)
                                    unsavedChangesDialog.pendingPath = ""
                                }
                                unsavedChangesDialog.close()
                            } else {
                                // No path - need to show save dialog first
                                // Store that we need to continue after save
                                window._pendingActionAfterSave = function() {
                                    if (unsavedChangesDialog.pendingPath) {
                                        window._openDroppedFile(unsavedChangesDialog.pendingPath)
                                        unsavedChangesDialog.pendingPath = ""
                                    }
                                }
                                window._openProjectSaveDialog("")
                                unsavedChangesDialog.close()
                            }
                        }
                    }
                }
            }
        }

        onRejected: {
            pendingPath = ""
        }
    }

    // Callback for continuing action after save dialog
    property var _pendingActionAfterSave: null

    // Centralized keyboard shortcuts component
    AppShortcuts {
        appWorkspaceController: window.appWorkspaceController
        settingsModel: window.settingsModel
        activeFocusItem: window.activeFocusItem
        onSaveProjectAsRequested: saveProjectDialog.openForSave("")
    }

    // Toast notification system
    ToastManager {
        id: toastManager
        anchors.top: parent.top
        anchors.right: parent.right
        anchors.topMargin: 60  // Below top bar
        anchors.rightMargin: Theme.spacingLg
        z: 9999  // Ensure toasts appear above everything
    }

    // Wire project save signals to toast notifications and pending actions
    Connections {
        target: window.appWorkspaceController || null
        function onProjectSaved(path) {
            toastManager.success(qsTr("Project saved"), "")
            // Execute any pending action that was waiting for save to complete
            if (window._pendingActionAfterSave) {
                var action = window._pendingActionAfterSave
                window._pendingActionAfterSave = null
                action()
            }
        }
        function onProjectSaveFailed(error) {
            toastManager.error(qsTr("Save failed: %1").arg(error), "")
            // Clear pending action on failure - don't proceed with load
            window._pendingActionAfterSave = null
        }
        function onDecode_warning(message) {
            // Split "Title: detail" format into title and detail
            var colonIndex = message.indexOf(": ")
            if (colonIndex > 0) {
                var title = message.substring(0, colonIndex)
                var detail = message.substring(colonIndex + 2)
                toastManager.warning(title, detail)
            } else {
                toastManager.warning(message, "")
            }
        }
    }

    // Wire license invalidation to toast notifications
    Connections {
        target: window.appWorkspaceController ? window.appWorkspaceController.licenseManager : null
        function onLicense_invalidated(title, detail) {
            toastManager.error(title, detail)
        }
    }

    // Welcome overlay shown on first boot
    WelcomeOverlay {
        id: welcomeOverlay
        anchors.centerIn: parent
        settingsModel: window.settingsModel
        languageController: window.languageController

        onClosed: {
            // After welcome is closed, trigger update check if enabled
            if (window.settingsModel && window.settingsModel.check_updates_on_startup() === true) {
                if (window.updateController) {
                    window.updateController.checkForUpdates()
                }
            }
        }
    }

    // Update available dialog
    UpdateAvailableDialog {
        id: updateAvailableDialog
        anchors.centerIn: parent
        updateController: window.updateController
        languageController: window.languageController
        settingsModel: window.settingsModel
    }

    // Handle update check completion
    // For manual checks (from settings), show popup immediately
    // For auto checks (on startup), just show the "New version!" indicator in menu bar
    Connections {
        target: window.updateController || null
        function onUpdateCheckComplete(hasUpdate, wasManualCheck) {
            if (hasUpdate && wasManualCheck) {
                updateAvailableDialog.open()
            }
        }
    }

    // Premium feature popup for Join tab
    PremiumFeaturePopup {
        id: joinPremiumPopup
        parent: Overlay.overlay
        anchors.centerIn: parent
        featureName: qsTr("Video Joining")
        featureDescription: qsTr("Add intros and outros to your videos without re-encoding hours of footage. We automatically convert only what's needed.")
        featureBenefits: [
            qsTr("Only incompatible files get converted - everything else joins instantly"),
            qsTr("Hours of footage + 10-second intro = seconds to export"),
            qsTr("No manual codec matching. Incompatible files are converted automatically to the right format."),
            qsTr("Use smart editing tools like speechless segment removal or transcript editing over multiple input files.")
        ]
        licenseManager: window.appWorkspaceController ? window.appWorkspaceController.licenseManager : null
        languageController: window.languageController

        onActivationSucceeded: {
            toastManager.success(qsTr("License activated!"), "")
            window.currentPageIndex = 3  // Switch to Join tab
            close()
        }
    }

    // Tutorial hint tooltip
    HintTooltip {
        id: hintTooltip
        parent: Overlay.overlay
        hint: window.appWorkspaceController ? window.appWorkspaceController.hintManager.currentHint : null
        searchRoot: contentLayout
        headerRoot: window.header
        footerRoot: mainTabs
        hintManager: window.appWorkspaceController ? window.appWorkspaceController.hintManager : null
        // Hide when export overlay is open (it has its own HintTooltip)
        // Must also check targetItem to avoid flash at (0,0) before target is found
        visible: !!hint && !!targetItem && window.appWorkspaceController && window.appWorkspaceController.mediaStatus === "ready" && !exportOverlay.visible
    }

    // Show welcome overlay or check for updates on startup
    Component.onCompleted: {
        if (window.settingsModel && window.settingsModel.is_first_boot()) {
            // First boot - show welcome overlay
            welcomeOverlay.open()
        } else if (window.settingsModel && window.settingsModel.check_updates_on_startup() === true) {
            // Not first boot, but update check enabled
            if (window.updateController) {
                window.updateController.checkForUpdates()
            }
        }
    }
}
