import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

Item {
    id: root
    objectName: "undoHistoryButton"

    // Required: the workspace controller providing undo history
    required property var workspaceController

    implicitWidth: historyButton.implicitWidth
    implicitHeight: historyButton.implicitHeight

    function closePopup(): void {
        historyPopup.close()
    }

    // Button that opens the history popup
    ToolButton {
        id: historyButton
        anchors.fill: parent
        text: qsTr("Undo history")
        enabled: !!(root.workspaceController && root.workspaceController.undoHistory && root.workspaceController.undoHistory.length > 0)

        background: Rectangle {
            color: historyButton.down ? Theme.background : (historyButton.hovered ? Theme.surfaceRaised : "transparent")
            radius: 4
        }

        contentItem: RowLayout {
            spacing: 4

            Image {
                source: Theme.iconUrl("arrow-back-up")
                sourceSize.width: 16
                sourceSize.height: 16
                Layout.alignment: Qt.AlignVCenter
            }

            Text {
                text: historyButton.text
                font: historyButton.font
                color: historyButton.enabled ? Theme.textPrimary : Theme.textSecondary
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
            }

            Text {
                text: "▾"
                font.pixelSize: 8
                color: Theme.textSecondary
                opacity: historyButton.enabled ? 1.0 : 0.5
                verticalAlignment: Text.AlignVCenter
            }
        }

        onClicked: historyPopup.open()
    }

    // Text metrics for measuring label widths
    TextMetrics {
        id: labelMetrics
        font.pixelSize: 13
    }

    // History dropdown popup
    Popup {
        id: historyPopup
        parent: historyButton
        y: historyButton.height + 4
        x: 0
        width: 150
        padding: 4

        onAboutToShow: {
            if (root.workspaceController && root.workspaceController.hintManager) {
                root.workspaceController.hintManager.dismissHint("undo_history")
            }

            // Calculate width based on longest label
            const history = root.workspaceController ? root.workspaceController.undoHistory : []
            if (!history || !history.length) {
                width = 150
                return
            }

            let maxWidth = 0
            for (let i = 0; i < history.length; i++) {
                labelMetrics.text = history[i].label || ""
                maxWidth = Math.max(maxWidth, labelMetrics.width)
            }
            // Add padding: 8 (left margin) + 16 (indicator) + 8 (spacing) + 8 (right margin) + 8 (popup padding)
            width = Math.max(150, Math.min(maxWidth + 48, 400))
        }

        background: Rectangle {
            color: Theme.surface
            border.color: Theme.surfaceBorder
            border.width: 1
            radius: 6
        }

        contentItem: Column {
            spacing: 2

            // Empty state
            Text {
                visible: !historyList.count
                text: qsTr("No history")
                color: Theme.textSecondary
                font.italic: true
                padding: 8
            }

            // History entries list
            ListView {
                id: historyList
                width: parent.width
                height: Math.min(contentHeight, 300)
                clip: true
                model: root.workspaceController ? root.workspaceController.undoHistory : []

                delegate: Rectangle {
                    id: entryDelegate
                    width: ListView.view.width
                    height: 32
                    radius: 4

                    required property var modelData
                    required property int index

                    // Expose parent references for ComponentBehavior: Bound
                    property var controller: root.workspaceController  // qmllint disable unqualified
                    property var popup: historyPopup  // qmllint disable unqualified

                    // Visual states: applied (past), current, redo-able (future)
                    property bool isApplied: modelData.isApplied
                    property bool isCurrent: modelData.isCurrent
                    property bool isRedoable: !modelData.isApplied && !modelData.isCurrent

                    color: {
                        if (entryArea.containsMouse) return Theme.surfaceRaised
                        if (isCurrent) return Theme.selectionHighlight
                        return "transparent"
                    }

                    RowLayout {
                        anchors.fill: parent
                        anchors.leftMargin: 8
                        anchors.rightMargin: 8
                        spacing: 8

                        // State indicator
                        Text {
                            Layout.preferredWidth: 16
                            text: {
                                if (entryDelegate.isCurrent) return "●"
                                if (entryDelegate.isApplied) return "✓"
                                return "○"  // redo-able
                            }
                            color: entryDelegate.isCurrent ? palette.highlight : Theme.textSecondary
                            opacity: entryDelegate.isRedoable ? 0.5 : 1.0
                            font.pixelSize: 12
                            horizontalAlignment: Text.AlignHCenter
                        }

                        // Action label
                        Text {
                            Layout.fillWidth: true
                            text: entryDelegate.modelData.label || qsTr("Unknown action")
                            color: entryDelegate.isRedoable ? Theme.textSecondary : Theme.textPrimary
                            elide: Text.ElideRight
                            font.pixelSize: 13
                        }
                    }

                    MouseArea {
                        id: entryArea
                        anchors.fill: parent
                        hoverEnabled: true
                        cursorShape: Qt.PointingHandCursor

                        onClicked: {
                            // Jump to this point in history
                            // The index in modelData is the stack position
                            // To make this entry the "current" one, we jump to index+1
                            // (since current_index = number of applied commands)
                            // Exception: clicking on current toggles it off (undo it)
                            if (entryDelegate.controller) {
                                const targetIndex = entryDelegate.isCurrent
                                    ? entryDelegate.modelData.index
                                    : entryDelegate.modelData.index + 1
                                entryDelegate.controller.jumpToHistoryIndex(targetIndex)
                            }
                            entryDelegate.popup.close()
                        }
                    }
                }
            }
        }
    }
}
