pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0

Item {
    id: root

    property bool enabled: true
    property string code: ""
    property bool includeAutonyms: true
    property bool showAutomatic: true
    property var settingsModel: null
    property string accessibleName: ""

    signal codeSelected(string code)

    readonly property var _allLanguages: TranscriptionLanguages.languages
    readonly property var _commonCodes: TranscriptionLanguages.commonCodes

    property var _recentCodes: []

    function _syncRecentsFromSettings() {
        if (!root.settingsModel || !root.settingsModel.recent_transcription_languages) {
            root._recentCodes = []
            return
        }
        root._recentCodes = root.settingsModel.recent_transcription_languages()
    }

    function _titleForCode(code) {
        if (code === null || code === undefined) return ""
        return TranscriptionLanguages.displayNameForCode(code, root.includeAutonyms)
    }

    function _isCommonCode(code) {
        for (let i = 0; i < root._commonCodes.length; i++) {
            if (root._commonCodes[i] === code) return true
        }
        return false
    }

    function _isRecentCode(code) {
        for (let i = 0; i < root._recentCodes.length; i++) {
            if (root._recentCodes[i] === code) return true
        }
        return false
    }

    function _entryForCode(code) {
        return TranscriptionLanguages.entryForCode(code)
    }

    function _normalizedSearchText(text) {
        return (text || "").toString().toLowerCase().trim()
    }

    function _matchesQuery(entry, query) {
        if (!entry) return false
        if (!query) return true
        const code = (entry.code || "").toString().toLowerCase()
        const name = (entry.name || "").toString().toLowerCase()
        const autonym = (entry.autonym || "").toString().toLowerCase()
        return code.indexOf(query) >= 0 || name.indexOf(query) >= 0 || autonym.indexOf(query) >= 0
    }

    function _sortedAllLanguagesWithoutAuto() {
        const result = []
        for (let i = 0; i < root._allLanguages.length; i++) {
            const entry = root._allLanguages[i]
            if (!entry || entry.code === "") continue
            result.push(entry)
        }
        result.sort(function(a, b) {
            const an = (a.name || "").toString()
            const bn = (b.name || "").toString()
            return an.localeCompare(bn)
        })
        return result
    }

    function _buildItems(query) {
        const q = root._normalizedSearchText(query)

        const items = []

        if (root.showAutomatic) {
            const autoEntry = root._entryForCode("")
            if (!q && autoEntry) {
                items.push({ kind: "header", title: qsTr("Automatic") })
                items.push({ kind: "language", entry: autoEntry })
            }
        }

        if (!q && root._recentCodes.length > 0) {
            items.push({ kind: "header", title: qsTr("Recent") })
            for (let i = 0; i < root._recentCodes.length; i++) {
                const entry = root._entryForCode(root._recentCodes[i])
                if (!entry || entry.code === "") continue
                items.push({ kind: "language", entry: entry })
            }
        }

        if (!q) {
            items.push({ kind: "header", title: qsTr("Common") })
            for (let i = 0; i < root._allLanguages.length; i++) {
                const entry = root._allLanguages[i]
                if (!entry) continue
                if (!root._isCommonCode(entry.code)) continue
                if (entry.code === "") continue
                items.push({ kind: "language", entry: entry })
            }

            items.push({ kind: "header", title: qsTr("All languages") })
            const allSorted = root._sortedAllLanguagesWithoutAuto()
            for (let i = 0; i < allSorted.length; i++) {
                items.push({ kind: "language", entry: allSorted[i] })
            }

            return items
        }

        // Search mode: only matching languages (no headers)
        for (let i = 0; i < root._allLanguages.length; i++) {
            const entry = root._allLanguages[i]
            if (!entry) continue
            if (!root._matchesQuery(entry, q)) continue
            items.push({ kind: "language", entry: entry })
        }

        if (items.length === 0) {
            items.push({ kind: "empty", title: qsTr("No matches") })
        }

        return items
    }

    function openPicker() {
        if (!root.enabled) return
        root._syncRecentsFromSettings()
        pickerSearch.text = ""
        pickerList.model = root._buildItems("")
        pickerPopup.open()
        pickerSearch.forceActiveFocus()
    }

    implicitHeight: 40
    Layout.fillWidth: true
    focus: true

    Accessible.role: Accessible.Button
    Accessible.name: root.accessibleName.length > 0 ? root.accessibleName : root._titleForCode(root.code)

    Keys.onPressed: function(event) {
        if (!root.enabled) return
        if (event.key === Qt.Key_Return || event.key === Qt.Key_Enter || event.key === Qt.Key_Space) {
            event.accepted = true
            root.openPicker()
        }
    }

    Rectangle {
        anchors.fill: parent
        radius: Theme.panelRadius - 4
        color: root.enabled ? Theme.surfaceRaised : Theme.surface
        border.width: 1
        border.color: Theme.surfaceBorder
    }

    MouseArea {
        anchors.fill: parent
        enabled: root.enabled
        cursorShape: Qt.PointingHandCursor
        onClicked: root.openPicker()
    }

    RowLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingSm
        spacing: Theme.spacingSm

        Label {
            Layout.fillWidth: true
            text: root._titleForCode(root.code)
            color: root.enabled ? Theme.textPrimary : Theme.textSecondary
            elide: Text.ElideRight
        }

        Label {
            text: "\u25BE"
            color: root.enabled ? Theme.textSecondary : Theme.textSecondary
        }
    }

    Popup {
        id: pickerPopup
        parent: Overlay.overlay
        modal: true
        focus: true
        padding: 0
        closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

        width: 520
        height: 560

        background: Rectangle {
            radius: Theme.panelRadius * 2
            color: Theme.surfaceStrong
            border.color: Theme.surfaceBorder
            border.width: 1
        }

        onAboutToShow: {
            const p = root.mapToItem(null, 0, root.height + Theme.spacingXs)
            pickerPopup.x = Math.max(Theme.spacingLg, Math.min(p.x, (root.Window.window ? root.Window.window.width : 9999) - pickerPopup.width - Theme.spacingLg))
            pickerPopup.y = Math.max(Theme.spacingLg, Math.min(p.y, (root.Window.window ? root.Window.window.height : 9999) - pickerPopup.height - Theme.spacingLg))
        }

        contentItem: ColumnLayout {
            anchors.fill: parent
            anchors.margins: Theme.spacingLg
            spacing: Theme.spacingMd

            TextField {
                id: pickerSearch
                Layout.fillWidth: true
                placeholderText: qsTr("Search languages…")
                selectByMouse: true
                onTextChanged: pickerList.model = root._buildItems(text)
            }

            ScrollView {
                Layout.fillWidth: true
                Layout.fillHeight: true
                clip: true

                ListView {
                    id: pickerList
                    width: parent.width
                    height: parent.height
                    model: root._buildItems("")
                    spacing: 2
                    clip: true

                    delegate: Item {
                        id: delegateRoot
                        required property var modelData

                        width: ListView.view.width
                        height: delegateRoot.modelData.kind === "header" ? 34 : (delegateRoot.modelData.kind === "empty" ? 44 : 44)

                        Rectangle {
                            anchors.fill: parent
                            radius: Theme.panelRadius - 4
                            color: delegateRoot.modelData.kind === "language" && mouseArea.containsMouse
                                ? Theme.surfaceRaised
                                : "transparent"
                        }

                        MouseArea {
                            id: mouseArea
                            anchors.fill: parent
                            hoverEnabled: true
                            enabled: delegateRoot.modelData.kind === "language"
                            cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor
                            onClicked: {
                                const entry = delegateRoot.modelData.entry
                                const newCode = entry ? entry.code : ""
                                root.code = newCode
                                if (root.settingsModel && root.settingsModel.add_recent_transcription_language) {
                                    root.settingsModel.add_recent_transcription_language(newCode)
                                }
                                root.codeSelected(newCode)
                                pickerPopup.close()
                            }
                        }

                        RowLayout {
                            anchors.fill: parent
                            anchors.margins: Theme.spacingSm
                            spacing: Theme.spacingSm

                            Label {
                                Layout.fillWidth: true
                                visible: delegateRoot.modelData.kind !== "header"
                                text: delegateRoot.modelData.kind === "language"
                                    ? TranscriptionLanguages.displayName(delegateRoot.modelData.entry, root.includeAutonyms)
                                    : (delegateRoot.modelData.kind === "empty" ? delegateRoot.modelData.title : "")
                                color: Theme.textPrimary
                                elide: Text.ElideRight
                            }

                            Label {
                                visible: delegateRoot.modelData.kind === "language" && root.code === delegateRoot.modelData.entry.code
                                text: "\u2713"
                                color: Theme.accent
                            }

                            Label {
                                Layout.fillWidth: true
                                visible: delegateRoot.modelData.kind === "header"
                                text: delegateRoot.modelData.title ?? ""
                                color: Theme.textSecondary
                                font.pixelSize: 12
                            }
                        }
                    }
                }
            }
        }
    }

    Component.onCompleted: root._syncRecentsFromSettings()

    Connections {
        target: root.settingsModel || null
        ignoreUnknownSignals: true
        function onRecent_transcription_languages_changed() {
            root._syncRecentsFromSettings()
        }
    }
}
