pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

ColumnLayout {
    id: root

    // Input: list of track objects with {index, label, color}
    property var tracks: []
    // Settings model to read default_track_mode ("all" or "first")
    property var settingsModel: null
    // Helper text shown below the checkboxes
    property string helperText: ""

    // Read-only: whether panel should be visible (>1 track)
    readonly property bool shouldShow: tracks && tracks.length > 1
    // Read-only: track count
    readonly property int trackCount: tracks ? tracks.length : 0

    // Emitted when user changes track selection
    signal selectionChanged()

    spacing: Theme.spacingXs
    Layout.fillWidth: true

    // Internal model for track selection state
    ListModel {
        id: selectionModel

        function initFromTracks(trackList, defaultMode) {
            clear()
            const selectAll = (defaultMode !== "first")
            for (let i = 0; i < trackList.length; i++) {
                // Use selection from track data if available, otherwise use default mode
                const trackSelected = (trackList[i].selected !== undefined)
                    ? trackList[i].selected
                    : (selectAll || (i === 0))
                append({
                    trackIndex: trackList[i].index,
                    label: trackList[i].label,
                    selected: trackSelected,
                    color: trackList[i].color || "#4c94ff"
                })
            }
        }

        function getSelectedIndices() {
            let indices = []
            for (let i = 0; i < count; i++) {
                if (get(i).selected) {
                    indices.push(get(i).trackIndex)
                }
            }
            return indices
        }
    }

    // Public function to get selected track indices
    function getSelectedIndices() {
        return selectionModel.getSelectedIndices()
    }

    // Public function to reinitialize from tracks
    function initFromTracks(trackList) {
        const mode = root.settingsModel ? root.settingsModel.default_track_mode() : "all"
        selectionModel.initFromTracks(trackList, mode)
    }

    // Initialize when tracks change
    onTracksChanged: {
        if (tracks && tracks.length > 0) {
            initFromTracks(tracks)
        }
    }

    Repeater {
        model: selectionModel
        delegate: AbstractButton {
            id: trackBtn
            required property int index
            required property int trackIndex
            required property string label
            required property bool selected
            required property string color
            Layout.fillWidth: true
            Layout.preferredHeight: 32
            checkable: true
            checked: selected
            onToggled: {
                selectionModel.setProperty(index, "selected", checked)
                root.selectionChanged()
            }

            background: Rectangle {
                color: trackBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04) : "transparent"
                radius: Theme.panelRadius - 4
            }

            contentItem: RowLayout {
                anchors.fill: parent
                anchors.leftMargin: Theme.spacingSm
                anchors.rightMargin: Theme.spacingSm
                spacing: Theme.spacingSm

                // Checkbox indicator with track color
                Rectangle {
                    Layout.preferredWidth: 16
                    Layout.preferredHeight: 16
                    radius: 3
                    color: trackBtn.checked ? trackBtn.color : "transparent"
                    border.width: 2
                    border.color: trackBtn.color

                    Label {
                        anchors.centerIn: parent
                        text: "✓"
                        font.pixelSize: 11
                        font.weight: Font.Bold
                        color: Theme.textOnAccent
                        visible: trackBtn.checked
                    }
                }

                Label {
                    Layout.fillWidth: true
                    text: trackBtn.label
                    color: Theme.textPrimary
                    font.pixelSize: 13
                    elide: Text.ElideRight
                }
            }
        }
    }

    Label {
        visible: root.helperText.length > 0
        text: root.helperText
        font.pixelSize: 12
        color: Theme.textSecondary
        opacity: 0.7
        wrapMode: Text.WordWrap
        Layout.fillWidth: true
    }
}
