pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import styles 1.0

/*
  OverviewTimeline: always zoomed-out timeline view.
  - Shows full-duration scrub area with playhead triangle and line
  - Displays cuts, selection, and loop overlays
  - Tick marks for time reference (no labels)
  - Hover tooltip shows current position and duration
*/
Item {
    id: root
    property real duration: 0
    property real currentTime: 0
    property var backend: null  // Timeline backend for live bindings
    property var appController: null
    property var loopRange: []
    // These are updated via Connections since QVariantList bindings don't auto-update
    property var cutRanges: []
    property var selectionRange: []
    // Viewport tracking for showing visible area of main timeline
    property real viewportStart: 0
    property real viewportEnd: duration
    signal seekRequested(real time)
    signal panRequested(real timeOffset)

    // Throttling for seek events to prevent flooding during rapid scrubbing
    property real pendingSeekTime: -1

    // Hover position for tooltip
    property real hoverTime: -1

    // Explicit connections to update properties when backend signals fire
    Connections {
        target: root.backend
        enabled: root.backend !== null
        function onCutsChanged() {
            root.cutRanges = root.backend ? root.backend.cutRanges : []
        }
        function onSelectionChanged() {
            root.selectionRange = root.backend ? root.backend.selectionRange : []
        }
        function onLoopRangeChanged() {
            root.loopRange = root.backend ? root.backend.loopRange : []
        }
    }

    onBackendChanged: {
        // Initialize when backend is set
        cutRanges = backend ? backend.cutRanges : []
        selectionRange = backend ? backend.selectionRange : []
        loopRange = backend ? backend.loopRange : []
    }

    function fmt(t) {
        const v = Math.max(0, Number(t) || 0)
        const m = Math.floor(v / 60)
        const s = Math.floor(v % 60)
        const cs = Math.floor((v * 100) % 100)
        const mm = String(m).padStart(2, '0')
        const ss = String(s).padStart(2, '0')
        const cc = String(cs).padStart(2, '0')
        return mm + ':' + ss + '.' + cc
    }

    implicitHeight: 28

    // Throttle seek emissions to ~60fps to prevent overwhelming audio engine
    Timer {
        id: seekThrottleTimer
        interval: 16  // ~60fps
        repeat: false
        onTriggered: {
            if (root.pendingSeekTime >= 0) {
                root.seekRequested(root.pendingSeekTime)
                root.pendingSeekTime = -1
            }
        }
    }

    Item {
        id: strip
        anchors.fill: parent

        readonly property real pct: root.duration > 0 ? Math.max(0, Math.min(1, root.currentTime / root.duration)) : 0
        readonly property real headX: pct * width

        // Background
        Rectangle {
            anchors.fill: parent
            radius: Theme.panelRadius
            color: Theme.surface
            border.color: Theme.surfaceBorder
        }

        // Tick marks canvas
        Canvas {
            id: tickCanvas
            anchors.fill: parent
            z: 1

            function tickStep(dur: real): real {
                var nice = [1, 2, 5, 10, 15, 30, 60, 120, 300, 600, 900, 1800, 3600]
                var target = dur / 10  // ~10 ticks across the overview
                for (var i = 0; i < nice.length; ++i) {
                    if (nice[i] >= target) return nice[i]
                }
                return nice[nice.length - 1]
            }

            onPaint: {
                var ctx = getContext("2d")
                ctx.reset()
                ctx.clearRect(0, 0, width, height)

                if (root.duration <= 0) return

                var step = tickStep(root.duration)
                ctx.strokeStyle = Qt.rgba(Theme.timelineTick.r, Theme.timelineTick.g, Theme.timelineTick.b, 0.3)
                ctx.lineWidth = 1

                for (var t = step; t < root.duration; t += step) {
                    var x = Math.round(t / root.duration * width) + 0.5
                    ctx.beginPath()
                    ctx.moveTo(x, 0)
                    ctx.lineTo(x, height)
                    ctx.stroke()
                }
            }

            Connections {
                target: root
                function onDurationChanged() { tickCanvas.requestPaint() }
            }
            onWidthChanged: requestPaint()
            Component.onCompleted: requestPaint()
        }

        // Cut overlays
        Repeater {
            model: root.cutRanges
            delegate: Rectangle {
                id: cutDelegate
                required property var modelData
                readonly property real startPct: root.duration > 0 ? modelData[0] / root.duration : 0
                readonly property real endPct: root.duration > 0 ? modelData[1] / root.duration : 0
                x: Math.round(startPct * strip.width)
                width: Math.max(1, Math.round((endPct - startPct) * strip.width))
                height: parent.height
                color: Theme.timelineCutOverlay
                border.color: Theme.timelineCutOverlayBorder
                border.width: 1
                radius: Theme.panelRadius
                z: 2
            }
        }

        // Selection overlay
        Rectangle {
            id: selectionOverlay
            readonly property bool hasSelection: root.selectionRange && root.selectionRange.length === 2 && root.duration > 0
            visible: hasSelection
            readonly property real startTime: hasSelection ? Math.min(root.selectionRange[0], root.selectionRange[1]) : 0
            readonly property real endTime: hasSelection ? Math.max(root.selectionRange[0], root.selectionRange[1]) : 0
            x: hasSelection ? Math.round(startTime / root.duration * strip.width) : 0
            width: hasSelection ? Math.max(1, Math.round((endTime - startTime) / root.duration * strip.width)) : 0
            height: strip.height
            color: Theme.selectionHighlight
            border.color: Theme.timelineSelectionBorder
            border.width: 1
            radius: Theme.panelRadius
            z: 3
        }

        // A/B loop overlay
        Rectangle {
            id: loopOverlay
            readonly property bool hasLoop: root.loopRange && root.loopRange.length === 2 && root.duration > 0
            visible: hasLoop
            color: Theme.timelineLoopFill
            border.color: Theme.timelineLoopBorder
            border.width: 1
            radius: Theme.panelRadius
            height: parent.height
            x: hasLoop ? Math.round(Math.min(root.loopRange[0], root.loopRange[1]) / root.duration * parent.width) : 0
            width: hasLoop ? Math.max(1, Math.round(Math.abs(root.loopRange[1] - root.loopRange[0]) / root.duration * parent.width)) : 0
            z: 4
        }

        // Viewport indicator - shows visible area of main timeline
        Rectangle {
            id: viewportIndicator
            objectName: "viewportIndicator"
            readonly property bool showViewport: root.duration > 0 &&
                (root.viewportEnd - root.viewportStart) < root.duration * 0.99
            visible: showViewport

            readonly property real startPct: root.duration > 0 ?
                Math.max(0, root.viewportStart) / root.duration : 0
            readonly property real endPct: root.duration > 0 ?
                Math.min(root.duration, root.viewportEnd) / root.duration : 1

            x: Math.round(startPct * strip.width)
            width: Math.max(2, Math.round((endPct - startPct) * strip.width))
            height: parent.height

            color: Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.04)
            border.color: Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.35)
            border.width: 1
            z: 7  // Above mouseArea (z:6) so handle can receive events

            // Drag handle at top - only this part is draggable for panning
            Rectangle {
                id: viewportHandle
                objectName: "viewportHandle"
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.top: parent.top
                height: 1
                color: Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.3)

                // Three-dot grip pattern
                Row {
                    anchors.horizontalCenter: parent.horizontalCenter
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.verticalCenterOffset: 2
                    spacing: 3
                    visible: parent.width > 24  // Hide dots if handle is too narrow
                    Repeater {
                        model: 3
                        Rectangle {
                            width: 3
                            height: 3
                            radius: 1.5
                            color: Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.6)
                        }
                    }
                }

                MouseArea {
                    anchors.fill: parent
                    cursorShape: pressed ? Qt.ClosedHandCursor : Qt.OpenHandCursor
                    property real dragStartX: 0
                    property real dragStartOffset: 0

                    onPressed: function(ev) {
                        dragStartX = ev.x + viewportIndicator.x
                        dragStartOffset = root.viewportStart
                        if (root.appController && root.appController.hintManager) {
                            root.appController.hintManager.dismissHint("viewport_indicator")
                        }
                    }
                    onPositionChanged: function(ev) {
                        if (!pressed) return
                        var deltaX = (ev.x + viewportIndicator.x) - dragStartX
                        var deltaTime = deltaX / strip.width * root.duration
                        root.panRequested(dragStartOffset + deltaTime)
                    }
                }

                // Extended hit area around grip dots for easier dragging
                MouseArea {
                    anchors.horizontalCenter: parent.horizontalCenter
                    anchors.verticalCenter: parent.verticalCenter
                    width: 22
                    height: 10
                    cursorShape: pressed ? Qt.ClosedHandCursor : Qt.OpenHandCursor
                    property real dragStartX: 0
                    property real dragStartOffset: 0

                    onPressed: function(ev) {
                        dragStartX = mapToItem(strip, ev.x, 0).x
                        dragStartOffset = root.viewportStart
                        if (root.appController && root.appController.hintManager) {
                            root.appController.hintManager.dismissHint("viewport_indicator")
                        }
                    }
                    onPositionChanged: function(ev) {
                        if (!pressed) return
                        var globalX = mapToItem(strip, ev.x, 0).x
                        var deltaX = globalX - dragStartX
                        var deltaTime = deltaX / strip.width * root.duration
                        root.panRequested(dragStartOffset + deltaTime)
                    }
                }
            }
        }

        // Playhead triangle (allow clipping at edges)
        Canvas {
            id: playheadTriangle
            anchors.top: parent.top
            x: strip.headX - width / 2
            width: 12
            height: 8
            z: 5
            clip: false
            // Repaint when theme changes
            property color accentColor: Theme.accent
            onAccentColorChanged: requestPaint()
            onPaint: {
                const ctx = getContext('2d')
                ctx.clearRect(0, 0, width, height)
                ctx.fillStyle = Qt.rgba(accentColor.r, accentColor.g, accentColor.b, 1)
                ctx.beginPath()
                ctx.moveTo(0, 0)
                ctx.lineTo(width, 0)
                ctx.lineTo(width / 2, height)
                ctx.closePath()
                ctx.fill()
            }
        }

        // Playhead line
        Rectangle {
            x: Math.round(strip.headX) - 1
            width: 2
            height: strip.height
            color: Theme.accent
            z: 5
        }

        // Mouse interaction area
        MouseArea {
            id: mouseArea
            anchors.fill: parent
            hoverEnabled: true
            z: 6

            onPressed: function(ev) {
                const pct = Math.max(0, Math.min(1, ev.x / strip.width))
                const time = pct * Math.max(0, root.duration)
                root.pendingSeekTime = time
                seekThrottleTimer.start()
            }
            onPositionChanged: function(ev) {
                // Update hover time for tooltip
                const pct = Math.max(0, Math.min(1, ev.x / strip.width))
                root.hoverTime = pct * Math.max(0, root.duration)

                if (!pressed) return
                const time = root.hoverTime
                root.pendingSeekTime = time
                seekThrottleTimer.start()
            }
            onExited: {
                root.hoverTime = -1
            }
            onWheel: function(wheel) {
                // Both horizontal and vertical scroll pan the main timeline
                var delta = wheel.angleDelta.x !== 0 ? wheel.angleDelta.x : wheel.angleDelta.y
                var visibleDuration = root.viewportEnd - root.viewportStart
                var panAmount = -delta / 120 * visibleDuration * 0.33  // 33% of visible width per notch
                root.panRequested(root.viewportStart + panAmount)
                wheel.accepted = true
            }

            ToolTip {
                id: hoverTooltip
                visible: mouseArea.containsMouse && root.duration > 0
                text: root.hoverTime >= 0
                      ? qsTr("%1 / %2").arg(root.fmt(root.hoverTime)).arg(root.fmt(root.duration))
                      : qsTr("%1 / %2").arg(root.fmt(root.currentTime)).arg(root.fmt(root.duration))
                delay: 100
                x: Math.max(0, Math.min(mouseArea.mouseX - width / 2, mouseArea.width - width))
                y: mouseArea.y - height - 2
            }
        }
    }
}
