pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import Qt5Compat.GraphicalEffects

import styles 1.0
import components.controls 1.0 as Controls

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    // Required properties
    required property string featureName
    required property string featureDescription
    required property var licenseManager
    required property var languageController

    // Optional customization
    property var featureBenefits: []
    property string purchaseButtonText: qsTr("View Plans & Pricing")

    // Signals
    signal activationSucceeded()
    signal activationFailed(string errorMessage)

    width: Math.min(parent ? parent.width * 0.9 : 520, 520)
    height: Math.min(parent ? parent.height * 0.9 : 650, 650)

    background: Rectangle {
        radius: Theme.panelRadius * 2
        color: Theme.surface
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    QtObject {
        id: state
        property bool isActivating: false
        property string errorMessage: ""
    }

    Connections {
        target: root.licenseManager
        enabled: state.isActivating  // Only listen when we initiated activation
        function onActivation_finished(errorCode) {
            state.isActivating = false
            if (errorCode === "") {
                licenseKeyField.text = ""
                root.activationSucceeded()
                root.close()
            } else {
                state.errorMessage = root.licenseErrorMessage(errorCode)
            }
        }
    }

    // Maps error codes from LicenseManager to localized user-facing strings
    function licenseErrorMessage(errorCode: string): string {
        switch (errorCode) {
            case "key_required":
                return qsTr("Please enter a license key")
            case "key_not_found":
                return qsTr("License key not found. Please check your key and try again.")
            case "key_expired":
                return qsTr("This license key has expired.")
            case "key_usage_limit":
                return qsTr("This license key has reached its activation limit. Please contact support at info@smartmediacutter.com.")
            case "server_error":
                return qsTr("Server error. Please try again later.")
            case "network_error":
                return qsTr("Network error. Please check your internet connection.")
            case "timeout":
                return qsTr("Connection timed out. Please try again.")
            case "invalid_response":
                return qsTr("Invalid server response. Please try again later.")
            case "token_save_failed":
                return qsTr("Failed to save license. Please try again.")
            default:
                return qsTr("Activation failed. Please try again.")
        }
    }

    contentItem: ColumnLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingXl
        spacing: Theme.spacingLg

        // Header with close button
        RowLayout {
            Layout.fillWidth: true
            Item { Layout.fillWidth: true }
            ToolButton {
                Accessible.name: qsTr("Close")
                text: "\u2715"
                font.pixelSize: 18
                implicitWidth: 36
                implicitHeight: 36
                onClicked: root.close()
            }
        }

        // Scrollable content
        Flickable {
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: width
            contentHeight: scrollContent.implicitHeight
            clip: true
            boundsBehavior: Flickable.StopAtBounds

            ColumnLayout {
                id: scrollContent
                width: parent.width
                spacing: Theme.spacingLg

                // Crown icon
                Item {
                    Layout.alignment: Qt.AlignHCenter
                    Layout.preferredWidth: 48
                    Layout.preferredHeight: 48

                    Image {
                        id: crownIcon
                        anchors.fill: parent
                        source: Theme.iconUrl("crown")
                        sourceSize.width: 48
                        sourceSize.height: 48
                        visible: false
                    }

                    ColorOverlay {
                        anchors.fill: parent
                        source: crownIcon
                        color: Theme.accent
                    }
                }

                // Title
                Label {
                    text: qsTr("Unlock %1").arg(root.featureName)
                    font.pixelSize: 24
                    font.weight: Font.DemiBold
                    color: Theme.textPrimary
                    Layout.alignment: Qt.AlignHCenter
                    wrapMode: Text.WordWrap
                    horizontalAlignment: Text.AlignHCenter
                    Layout.fillWidth: true
                }

                // Description
                Label {
                    text: root.featureDescription
                    font.pixelSize: 14
                    color: Theme.textSecondary
                    wrapMode: Text.WordWrap
                    horizontalAlignment: Text.AlignHCenter
                    Layout.fillWidth: true
                    lineHeight: 1.3
                }

                // Benefits list (if provided)
                ColumnLayout {
                    visible: root.featureBenefits.length > 0
                    Layout.fillWidth: true
                    Layout.topMargin: Theme.spacingSm
                    spacing: Theme.spacingXs

                    Repeater {
                        model: root.featureBenefits
                        delegate: RowLayout {
                            id: benefitRow
                            required property string modelData
                            Layout.fillWidth: true
                            spacing: Theme.spacingSm

                            Label {
                                text: "\u2713"
                                color: Theme.accent
                                font.pixelSize: 14
                                font.weight: Font.Bold
                            }
                            Label {
                                text: benefitRow.modelData
                                color: Theme.textPrimary
                                font.pixelSize: 13
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }
                        }
                    }
                }

                // Purchase button
                Controls.PrimaryButton {
                    text: root.purchaseButtonText
                    Layout.fillWidth: true
                    Layout.preferredHeight: 44
                    Layout.topMargin: Theme.spacingSm
                    onClicked: {
                        var lang = root.languageController ? root.languageController.currentLanguage : "en"
                        Qt.openUrlExternally(root.licenseManager.localizedPricingUrl(lang))
                    }
                }

                // Divider with text
                RowLayout {
                    Layout.fillWidth: true
                    Layout.topMargin: Theme.spacingMd
                    spacing: Theme.spacingMd

                    Rectangle {
                        Layout.fillWidth: true
                        Layout.preferredHeight: 1
                        color: Theme.surfaceBorder
                    }
                    Label {
                        text: qsTr("Already have a license?")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                    }
                    Rectangle {
                        Layout.fillWidth: true
                        Layout.preferredHeight: 1
                        color: Theme.surfaceBorder
                    }
                }

                // License key input section
                ColumnLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingSm

                    TextField {
                        id: licenseKeyField
                        Layout.fillWidth: true
                        placeholderText: qsTr("Enter your license key")
                        enabled: !state.isActivating
                        onTextChanged: state.errorMessage = ""
                    }

                    Controls.SecondaryButton {
                        text: state.isActivating ? qsTr("Activating...") : qsTr("Activate License")
                        Layout.fillWidth: true
                        enabled: !state.isActivating && licenseKeyField.text.trim().length > 0

                        onClicked: {
                            state.isActivating = true
                            state.errorMessage = ""
                            root.licenseManager.activateLicense(licenseKeyField.text.trim())
                        }
                    }

                    // Error message
                    Label {
                        visible: state.errorMessage.length > 0
                        text: state.errorMessage
                        color: Theme.statusDanger
                        font.pixelSize: 12
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }
                }
            }
        }
    }
}
