pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Layouts 1.15
import components.layout 1.0
import components.media 1.0
import styles 1.0

/*
  TimelineSection: Combines OverviewTimeline + AudioTimeline.

  Usage:
    TimelineSection {
        Layout.fillWidth: true
        timelineBackend: root.timelineBackend
        appController: root.appController
        showOverviewTimeline: viewPrefs.prefs.showOverviewTimeline
        showAudioTimeline: viewPrefs.prefs.showAudioTimeline
        timelineReady: root.timelineReady
        emptyStateText: root.timelineEmptyText
    }
*/
ColumnLayout {
    id: root

    required property var timelineBackend
    required property var appController

    property bool showOverviewTimeline: true
    property bool showAudioTimeline: false

    property int waveformTrackHeight: 76
    property bool showWaveforms: true
    property string emptyStateText: ""
    property bool timelineReady: true

    spacing: 0

    readonly property real mediaDuration: appController ? appController.mediaDuration : 0

    // Overview timeline (mini)
    Card {
        Layout.fillWidth: true
        contentPadding: Theme.spacingMd
        visible: root.showOverviewTimeline

        OverviewTimeline {
            anchors.fill: parent
            duration: root.mediaDuration
            currentTime: root.timelineBackend ? root.timelineBackend.playheadTime : 0
            backend: root.timelineBackend
            appController: root.appController
            loopRange: root.appController ? root.appController.loopRange : []

            // Viewport tracking (uses audioTimeline's local view state)
            viewportStart: audioTimeline.timeOffset
            viewportEnd: audioTimeline.visibleDuration > 0 ?
                audioTimeline.timeOffset + audioTimeline.visibleDuration : root.mediaDuration

            onSeekRequested: function(t) {
                if (root.appController) root.appController.seekToTime(t)
            }
            onPanRequested: function(offset) {
                var clampedOffset = audioTimeline.clampTimeOffset(offset)
                audioTimeline.timeOffset = clampedOffset
                audioTimeline.requestViewUpdate()
            }
        }
    }

    // Audio timeline (detailed)
    TimelineCard {
        id: timelineCard
        Layout.fillWidth: true
        Layout.preferredHeight: audioTimeline.implicitHeight
        visible: root.showAudioTimeline

        AudioTimeline {
            id: audioTimeline
            objectName: "audioTimeline"
            anchors.fill: parent
            backend: root.timelineBackend
            analysisController: root.appController ? root.appController.analysisController : null
            hintManager: root.appController ? root.appController.hintManager : null
            enabled: root.timelineReady
            emptyStateText: root.emptyStateText
            trackHeight: root.waveformTrackHeight
            showWaveforms: root.showWaveforms
            onPlayheadChanged: function(time) {
                if (root.appController) root.appController.seekToTime(time)
            }
        }
    }

    // Public API for centering
    function centerOnRange(startTime, endTime) {
        audioTimeline.centerOnRange(startTime, endTime)
    }

    function fitToDuration() {
        audioTimeline.fitToDuration()
    }
}
