pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Shapes 1.15

import styles 1.0
import components.controls 1.0

Popup {
    id: root
    parent: Overlay.overlay
    anchors.centerIn: parent
    // Large popup - 80% of screen
    implicitWidth: parent ? parent.width * 0.8 : 800
    implicitHeight: Math.min(400, parent ? parent.height * 0.8 : 600)
    modal: true
    focus: true
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    property real initialValue: -22
    property real tempDb: initialValue

    // Controller and track selection for fetching waveform data
    property var analysisController: null
    property var selectedTrackIndices: []

    // Timeline backend for zoomed view
    property var timelineBackend: null

    // View mode: false = global (full waveform), true = zoomed (timeline view)
    property bool zoomedView: false

    // Waveform data: amplitude values 0.0-1.0
    property var waveformData: []

    signal applied(real value)

    function refreshWaveform(): void {
        if (!analysisController || waveformContainer.width <= 0) {
            waveformData = []
            return
        }
        let indices = selectedTrackIndices.length > 0 ? selectedTrackIndices : [0]
        let w = Math.round(waveformContainer.width)

        if (zoomedView && timelineBackend) {
            // Get visible time range from timeline
            let startTime = timelineBackend.timeOffset
            let visibleDuration = timelineBackend.secondsPerPixel * waveformContainer.width
            let endTime = startTime + visibleDuration
            waveformData = analysisController.getThresholdPickerWaveform(indices, w, startTime, endTime)
        } else {
            // Global view - full waveform
            waveformData = analysisController.getThresholdPickerWaveform(indices, w)
        }
    }

    Overlay.modal: Rectangle {
        color: Qt.rgba(0, 0, 0, 0.7)
    }

    background: Rectangle {
        color: Theme.surface
        radius: Theme.panelRadius
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    onOpened: {
        tempDb = initialValue
        refreshWaveform()
    }

    onZoomedViewChanged: {
        if (opened) {
            refreshWaveform()
        }
    }

    ColumnLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingLg
        spacing: Theme.spacingMd

        Label {
            text: qsTr("Set Silence Threshold")
            font.pixelSize: 18
            font.weight: Font.Bold
            color: Theme.textPrimary
        }

        Label {
            text: qsTr("Drag the line up or down. Audio below the line (red) will be considered silence.")
            color: Theme.textSecondary
            font.pixelSize: 14
            Layout.fillWidth: true
            wrapMode: Text.WordWrap
        }

        // Visual Picker Area
        Item {
            id: waveformContainer
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true

            // Convert dB to normalized threshold (0-1 amplitude)
            readonly property real thresholdNorm: Math.pow(10, root.tempDb / 20)
            // Threshold Y position in pixels (0 is top, height is bottom)
            readonly property real thresholdY: height * (1 - thresholdNorm)

            Rectangle {
                anchors.fill: parent
                color: Theme.timelineBackground
                radius: Theme.panelRadius
                border.color: Theme.surfaceBorder
            }

            // Generate waveform path from amplitude data (0-1 values)
            function generateWaveformPath(data, w, h) {
                if (!data || data.length === 0 || h <= 0 || w <= 0) {
                    return ""
                }

                var parts = []
                parts.push("M 0 " + h)  // Start at bottom-left

                var xScale = w / data.length

                // Draw waveform: amplitude 0 = bottom, amplitude 1 = top
                for (var i = 0; i < data.length; i++) {
                    var x = i * xScale
                    var amplitude = data[i] || 0
                    // Y coordinate: 0 at top (loud), h at bottom (silent)
                    var y = h * (1 - amplitude)
                    parts.push("L " + x + " " + y)
                }

                parts.push("L " + w + " " + h)  // End at bottom-right
                parts.push("Z")  // Close path

                return parts.join(" ")
            }

            // Silence layer (red, below threshold) - full waveform with red tint
            Shape {
                id: silenceWaveform
                anchors.fill: parent
                anchors.margins: 1
                antialiasing: true
                layer.enabled: true
                layer.smooth: true

                ShapePath {
                    strokeWidth: 0
                    strokeColor: "transparent"
                    fillColor: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, 0.4)

                    PathSvg {
                        path: waveformContainer.generateWaveformPath(root.waveformData, silenceWaveform.width, silenceWaveform.height)
                    }
                }
            }

            // Audio layer (accent color, above threshold) - clipped to above threshold line
            Item {
                anchors.fill: parent
                anchors.margins: 1
                clip: true

                // Clip rectangle - only show area above threshold
                Item {
                    anchors.left: parent.left
                    anchors.right: parent.right
                    anchors.top: parent.top
                    height: waveformContainer.thresholdY
                    clip: true

                    Shape {
                        id: audioWaveform
                        width: parent.parent.width
                        height: parent.parent.height
                        y: 0
                        antialiasing: true
                        layer.enabled: true
                        layer.smooth: true

                        ShapePath {
                            strokeWidth: 0
                            strokeColor: "transparent"

                            fillGradient: LinearGradient {
                                y1: 0
                                y2: audioWaveform.height
                                GradientStop { position: 0.0; color: Qt.lighter(Theme.accent, 1.4) }
                                GradientStop { position: 1.0; color: Theme.accent }
                            }

                            PathSvg {
                                path: waveformContainer.generateWaveformPath(root.waveformData, audioWaveform.width, audioWaveform.height)
                            }
                        }
                    }
                }
            }

            // Threshold line
            Rectangle {
                id: thresholdLine
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.leftMargin: 1
                anchors.rightMargin: 1
                y: waveformContainer.thresholdY
                height: 2
                color: Theme.textPrimary

                // Drag handle visual
                Rectangle {
                    anchors.horizontalCenter: parent.horizontalCenter
                    anchors.verticalCenter: parent.verticalCenter
                    width: 60
                    height: 20
                    radius: 10
                    color: Theme.surfaceRaised
                    border.color: Theme.textPrimary
                    border.width: 1

                    // Grip lines
                    Row {
                        anchors.centerIn: parent
                        spacing: 3
                        Repeater {
                            model: 3
                            Rectangle {
                                width: 2
                                height: 10
                                radius: 1
                                color: Theme.textSecondary
                            }
                        }
                    }
                }
            }

            // Interactive overlay
            MouseArea {
                anchors.fill: parent
                anchors.margins: 1
                cursorShape: Qt.SplitVCursor

                function updateThreshold(mouseY: real): void {
                    var h = height
                    // Normalize 0..1 (0 is bottom of graph for amplitude)
                    var linear = 1.0 - (mouseY / h)
                    // Clamp
                    linear = Math.max(0.001, Math.min(1.0, linear))

                    // Convert to dB
                    var db = 20 * Math.log10(linear)

                    // Clamp to slider range
                    db = Math.max(-60, Math.min(0, db))
                    root.tempDb = db
                }

                onPressed: (mouse) => updateThreshold(mouse.y)
                onPositionChanged: (mouse) => {
                    if (pressed) updateThreshold(mouse.y)
                }
            }

            // Top-right controls: view toggle and dB label
            Row {
                anchors.right: parent.right
                anchors.top: parent.top
                anchors.margins: Theme.spacingMd
                spacing: Theme.spacingSm

                // View mode toggle
                Rectangle {
                    width: viewToggleRow.implicitWidth + Theme.spacingMd
                    height: 32
                    color: Theme.surfaceRaised
                    radius: 6
                    border.color: Theme.surfaceBorder
                    visible: root.timelineBackend !== null

                    Row {
                        id: viewToggleRow
                        anchors.centerIn: parent
                        spacing: 2

                        Rectangle {
                            width: globalLabel.implicitWidth + Theme.spacingSm
                            height: 24
                            radius: 4
                            color: !root.zoomedView ? Theme.accent : "transparent"

                            Label {
                                id: globalLabel
                                anchors.centerIn: parent
                                text: qsTr("Global")
                                font.pixelSize: 12
                                color: !root.zoomedView ? Theme.textOnAccent : Theme.textSecondary
                            }

                            MouseArea {
                                anchors.fill: parent
                                onClicked: root.zoomedView = false
                                cursorShape: Qt.PointingHandCursor
                            }
                        }

                        Rectangle {
                            width: zoomedLabel.implicitWidth + Theme.spacingSm
                            height: 24
                            radius: 4
                            color: root.zoomedView ? Theme.accent : "transparent"

                            Label {
                                id: zoomedLabel
                                anchors.centerIn: parent
                                text: qsTr("Zoomed")
                                font.pixelSize: 12
                                color: root.zoomedView ? Theme.textOnAccent : Theme.textSecondary
                            }

                            MouseArea {
                                anchors.fill: parent
                                onClicked: root.zoomedView = true
                                cursorShape: Qt.PointingHandCursor
                            }
                        }
                    }
                }

                // Floating Value Label
                Rectangle {
                    width: dbLabel.implicitWidth + Theme.spacingLg
                    height: 32
                    color: Theme.surfaceRaised
                    radius: 6
                    border.color: Theme.surfaceBorder

                    Label {
                        id: dbLabel
                        anchors.centerIn: parent
                        text: root.tempDb.toFixed(1) + " dB"
                        font.pixelSize: 14
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }
                }
            }

            // Empty state
            Label {
                anchors.centerIn: parent
                text: qsTr("No waveform data available")
                color: Theme.textSecondary
                font.pixelSize: 14
                visible: !root.waveformData || root.waveformData.length === 0
            }
        }

        RowLayout {
            Layout.fillWidth: true
            Layout.alignment: Qt.AlignRight
            spacing: Theme.spacingSm

            Button {
                text: qsTr("Cancel")
                onClicked: root.close()
            }

            PrimaryButton {
                text: qsTr("Apply")
                onClicked: {
                    root.applied(root.tempDb)
                    root.close()
                }
            }
        }
    }
}
