pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import Qt5Compat.GraphicalEffects

import styles 1.0
import components.controls 1.0
import components.forms 1.0 as Forms
import components.layout 1.0

Item {
    id: root
    // Ensure nothing can render outside this panel's bounds
    clip: true
    required property var appController
    property var settingsModel: null
    readonly property bool mediaReady: appController.mediaStatus === "ready"
    readonly property var analysisController: appController.analysisController
    property string selectedTool: ""
    readonly property bool showingDetail: selectedTool.length > 0
    property string lastError: ""

    // Notify hint manager when tool panels open/close
    onSelectedToolChanged: {
        if (root.appController && root.appController.hintManager) {
            root.appController.hintManager.set_speechless_panel_open(selectedTool === "speechless")
            root.appController.hintManager.set_silence_panel_open(selectedTool === "silence")
            root.appController.hintManager.set_black_frame_panel_open(selectedTool === "black")
            if (selectedTool === "speechless") {
                root.appController.hintManager.dismissHint("speechless_tool")
            }
        }
    }

    implicitWidth: 320
    implicitHeight: 640


    // Listen for analysis errors
    Connections {
        target: root.analysisController
        function onErrorOccurred(message) {
            root.lastError = message
            errorTimer.restart()
        }
    }

    // Notify hint manager when black visualization visibility changes
    Connections {
        target: root.analysisController
        function onBlackVisualizationVisibleChanged() {
            if (root.appController && root.appController.hintManager) {
                root.appController.hintManager.set_black_visualization_visible(
                    root.analysisController.blackVisualizationVisible
                )
            }
        }
    }

    Timer {
        id: errorTimer
        interval: 8000
        onTriggered: root.lastError = ""
    }

    readonly property var toolGroups: [
        {
            title: qsTr("Quick Trim"),
            tools: [
                {
                    key: "cutBefore",
                    title: qsTr("Cut Before Playhead"),
                    description: qsTr("Cut everything before the playhead."),
                    immediate: true,
                },
                {
                    key: "cutAfter",
                    title: qsTr("Cut After Playhead"),
                    description: qsTr("Cut everything after the playhead."),
                    immediate: true,
                },
                {
                    key: "invert",
                    title: qsTr("Invert Cuts"),
                    description: qsTr("Swap cut and kept segments."),
                    immediate: true,
                }
            ]
        },
        {
            title: qsTr("Auto-Detection"),
            tools: [
                {
                    key: "speechless",
                    title: qsTr("Speechless Detection"),
                    description: qsTr("Detect speech and cut parts without speech."),
                },
                {
                    key: "silence",
                    title: qsTr("Silence Detection"),
                    description: qsTr("Detect and cut all silent regions."),
                },
                {
                    key: "black",
                    title: qsTr("Black Frame Detection"),
                    description: qsTr("Detect and cut black sections of video."),
                },
            ]
        }
    ]

    function toolFor(key) {
        for (let g = 0; g < toolGroups.length; ++g) {
            let group = toolGroups[g]
            for (let i = 0; i < group.tools.length; ++i) {
                if (group.tools[i].key === key) {
                    return group.tools[i]
                }
            }
        }
        return null
    }

    Frame {
        anchors.fill: parent
        // Control children can overflow if text isn't width-constrained; keep it clipped.
        clip: true
        background: Rectangle {
            radius: Theme.panelRadius
            color: Theme.surface
            border.color: Theme.surfaceBorder
        }

        // Anchor children instead of overriding contentItem to ensure
        // padding/margins behave predictably and avoid width overflow.
        StackLayout {
            id: viewStack
            anchors.fill: parent
            currentIndex: root.showingDetail ? 1 : 0

            // Main List View
            ColumnLayout {
                spacing: 0 // Spacing handled by padding/margins of items
                Layout.fillWidth: true
                Layout.fillHeight: true

                SidebarHeader {
                    title: qsTr("Tools")
                }

                ScrollView {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    clip: true
                    contentWidth: availableWidth

                    ColumnLayout {
                        width: parent.width
                        spacing: 0

                        Repeater {
                            model: root.toolGroups
                            delegate: ColumnLayout {
                                id: groupDelegate
                                required property var modelData
                                required property int index
                                Layout.fillWidth: true
                                spacing: Theme.spacingSm

                                // Group header
                                Label {
                                    Layout.fillWidth: true
                                    Layout.topMargin: groupDelegate.index === 0 ? Theme.spacingMd : Theme.spacingLg
                                    Layout.leftMargin: Theme.spacingMd
                                    Layout.rightMargin: Theme.spacingMd
                                    Layout.bottomMargin: Theme.spacingXs
                                    text: groupDelegate.modelData.title
                                    font.pixelSize: 12
                                    font.weight: Font.DemiBold
                                    color: Theme.textSecondary
                                    opacity: 0.8
                                }

                                // Tools in this group
                                Repeater {
                                    model: groupDelegate.modelData.tools
                                    delegate: AbstractButton {
                                        id: control
                                        required property var modelData
                                        objectName: modelData.key + "ToolItem"
                                        Layout.fillWidth: true
                                        Layout.preferredHeight: 72
                                        Layout.leftMargin: Theme.spacingMd
                                        Layout.rightMargin: Theme.spacingMd

                                        onClicked: {
                                            if (modelData.immediate) {
                                                if (modelData.key === "invert") {
                                                    if (root.analysisController) {
                                                        root.analysisController.runInvertCuts()
                                                    }
                                                } else if (modelData.key === "cutBefore") {
                                                    if (root.appController) {
                                                        root.appController.cutBeforePlayhead()
                                                    }
                                                } else if (modelData.key === "cutAfter") {
                                                    if (root.appController) {
                                                        root.appController.cutAfterPlayhead()
                                                    }
                                                }
                                            } else {
                                                root.selectedTool = modelData.key
                                            }
                                        }

                                        background: Rectangle {
                                            radius: Theme.panelRadius
                                            color: control.hovered
                                                ? Theme.surfaceRaised
                                                : Qt.rgba(Theme.surfaceRaised.r, Theme.surfaceRaised.g, Theme.surfaceRaised.b, 0)
                                            border.color: control.hovered
                                                ? Theme.surfaceBorder
                                                : Qt.rgba(Theme.surfaceBorder.r, Theme.surfaceBorder.g, Theme.surfaceBorder.b, 0)
                                            border.width: 1

                                            Behavior on color { ColorAnimation { duration: 100 } }
                                        }

                                        contentItem: RowLayout {
                                            anchors.fill: parent
                                            anchors.margins: Theme.spacingMd
                                            spacing: Theme.spacingMd

                                            ColumnLayout {
                                                Layout.fillWidth: true
                                                spacing: 4

                                                Label {
                                                    Layout.fillWidth: true
                                                    text: control.modelData.title
                                                    font.pixelSize: 15
                                                    font.weight: Font.DemiBold
                                                    color: Theme.textPrimary
                                                    elide: Label.ElideRight
                                                }

                                                Label {
                                                    Layout.fillWidth: true
                                                    text: control.modelData.description
                                                    font.pixelSize: 13
                                                    color: Theme.textSecondary
                                                    elide: Label.ElideRight
                                                    maximumLineCount: 2
                                                    wrapMode: Text.WordWrap
                                                }
                                            }

                                            Image {
                                                id: chevronIcon
                                                Layout.preferredWidth: 16
                                                Layout.preferredHeight: 16
                                                source: Theme.iconUrl("chevron-right")
                                                rotation: 180
                                                fillMode: Image.PreserveAspectFit
                                                visible: false  // Hidden, ColorOverlay renders it
                                            }

                                            ColorOverlay {
                                                Layout.preferredWidth: 16
                                                Layout.preferredHeight: 16
                                                source: chevronIcon
                                                color: Theme.textPrimary
                                                opacity: 0.5
                                                visible: !control.modelData.immediate
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        Item { Layout.fillHeight: true }
                    }
                }
            }

            // Detail View
            ColumnLayout {
                id: detailView
                spacing: 0
                Layout.fillWidth: true
                Layout.fillHeight: true
                property var toolDefinition: root.toolFor(root.selectedTool)

                // Header
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 64
                    color: "transparent"

                    RowLayout {
                        anchors.fill: parent
                        anchors.leftMargin: Theme.spacingMd
                        anchors.rightMargin: Theme.spacingLg
                        spacing: Theme.spacingSm

                        IconButton {
                            objectName: "analysisBackButton"
                            iconSource: Theme.iconUrl("chevron-left")
                            Layout.preferredWidth: 36
                            Layout.preferredHeight: 36
                            onClicked: root.selectedTool = ""
                        }

                        Label {
                            Layout.fillWidth: true
                            text: detailView.toolDefinition ? detailView.toolDefinition.title : ""
                            color: Theme.textPrimary
                            font.pixelSize: 18
                            font.weight: Font.DemiBold
                            verticalAlignment: Text.AlignVCenter
                            elide: Label.ElideRight
                        }
                    }

                    Rectangle {
                        anchors.bottom: parent.bottom
                        anchors.left: parent.left
                        anchors.right: parent.right
                        height: 1
                        color: Theme.surfaceBorder
                    }
                }

                // Content
                ScrollView {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    clip: true
                    contentWidth: availableWidth

                    ColumnLayout {
                        width: parent.width
                        spacing: Theme.spacingLg

                        // Content Padding wrapper
                        Item {
                            Layout.fillWidth: true
                            Layout.preferredHeight: contentCol.height

                            ColumnLayout {
                                id: contentCol
                                anchors.left: parent.left
                                anchors.right: parent.right
                                anchors.top: parent.top
                                anchors.margins: Theme.spacingLg
                                spacing: Theme.spacingLg

                                Label {
                                    text: detailView.toolDefinition ? detailView.toolDefinition.description : ""
                                    wrapMode: Text.WordWrap
                                    color: Theme.textSecondary
                                    font.pixelSize: 14
                                    Layout.fillWidth: true
                                }

                                Loader {
                                    Layout.fillWidth: true
                                    sourceComponent: root.selectedTool === "silence"
                                        ? silenceToolView
                                        : root.selectedTool === "black"
                                            ? blackToolView
                                            : root.selectedTool === "speechless"
                                                ? speechlessToolView
                                                : undefined

                                    onLoaded: {
                                        if (item) {
                                            item.appController = Qt.binding(() => root.appController)
                                            item.mediaReady = Qt.binding(() => root.mediaReady)
                                            item.analysisController = Qt.binding(() => root.analysisController)
                                        }
                                    }
                                }

                                // Error display
                                Rectangle {
                                    Layout.fillWidth: true
                                    Layout.preferredHeight: errorLabel.implicitHeight + Theme.spacingMd * 2
                                    visible: root.lastError.length > 0
                                    color: Theme.statusDanger
                                    opacity: 0.15
                                    radius: Theme.panelRadius - 2

                                    Label {
                                        id: errorLabel
                                        anchors.fill: parent
                                        anchors.margins: Theme.spacingMd
                                        text: root.lastError
                                        wrapMode: Text.WordWrap
                                        color: Theme.statusDanger
                                        font.pixelSize: 13
                                    }
                                }
                            }
                        }

                        Item { Layout.fillHeight: true }
                    }
                }
            }
        }
    }

    Component {
        id: silenceToolView
        Item {
            id: silenceRoot
            property var appController: null
            property bool mediaReady: false
            property var analysisController: null

            ColumnLayout {
                anchors.fill: parent
                spacing: Theme.spacingSm

            // DEFINE SILENCE section
            Label {
                text: qsTr("DEFINE SILENCE")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Volume below")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingXs

                    LabeledSlider {
                        id: silenceThresholdSlider
                        Layout.fillWidth: true
                        from: -60
                        to: 0
                        stepSize: 1
                        value: -22
                        suffix: " dB"
                    }

                    IconButton {
                        iconSource: Theme.iconUrl("color-picker")
                        ToolTip.text: qsTr("Visually pick threshold")
                        ToolTip.visible: hovered
                        Layout.preferredWidth: 28
                        Layout.preferredHeight: 28
                        onClicked: thresholdPicker.open()
                    }
                }

                Label {
                    text: qsTr("For more than")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: silenceMinDurationSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 10
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Audio below threshold for longer than minimum is marked as silence")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // DEFINE KEPT SEGMENTS section
            Label {
                text: qsTr("DEFINE AUDIO")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Minimum length")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: silenceAudioMinLengthSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 50
                    stepSize: 1
                    value: 0
                    decimals: 2
                    displayScale: 0.01
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Segments shorter than this are merged into adjacent cuts")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // PADDING section
            Label {
                text: qsTr("PADDING")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Pre-audio")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: silencePaddingBeforeAudioSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 1
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }

                Label {
                    text: qsTr("Post-audio")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: silencePaddingAfterAudioSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 1
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Extra time kept around detected audio")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // TRACKS section - only show if multiple tracks
            Label {
                text: qsTr("TRACKS")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
                visible: silenceTrackPanel.shouldShow
            }

            Forms.TrackSelectionPanel {
                id: silenceTrackPanel
                Layout.fillWidth: true
                visible: shouldShow
                tracks: silenceRoot.analysisController ? silenceRoot.analysisController.tracks : []
                settingsModel: root.settingsModel
                helperText: qsTr("Select which audio track to analyze for silence")
            }

            CheckBox {
                id: silenceClearExisting
                Layout.fillWidth: true
                text: qsTr("Clear existing cuts before applying")
                checked: false
                enabled: silenceRoot.mediaReady

                contentItem: Label {
                    text: silenceClearExisting.text
                    font: silenceClearExisting.font
                    color: Theme.textPrimary
                    opacity: silenceClearExisting.enabled ? 1.0 : 0.5
                    verticalAlignment: Text.AlignVCenter
                    leftPadding: silenceClearExisting.indicator.width + silenceClearExisting.spacing
                    elide: Text.ElideRight
                }
            }

            PrimaryButton {
                Layout.fillWidth: true
                text: silenceRoot.analysisController && silenceRoot.analysisController.busy
                      ? qsTr("Analyzing…")
                      : qsTr("Cut Silent Regions")
                enabled: silenceRoot.mediaReady && silenceRoot.analysisController && !silenceRoot.analysisController.busy
                onClicked: {
                    if (!silenceRoot.analysisController) {
                        return
                    }
                    // Get first selected track, or default to 0
                    let selectedTracks = silenceTrackPanel.getSelectedIndices()
                    let trackIndex = selectedTracks.length > 0 ? selectedTracks[0] : 0
                    silenceRoot.analysisController.runSilenceDetection(
                        silenceThresholdSlider.value,
                        silencePaddingBeforeAudioSlider.value / 10.0,
                        silencePaddingAfterAudioSlider.value / 10.0,
                        silenceMinDurationSlider.value / 10.0,
                        silenceAudioMinLengthSlider.value / 100.0,
                        trackIndex,
                        silenceClearExisting.checked
                    )
                }
            }

            ProgressBar {
                Layout.fillWidth: true
                visible: silenceRoot.analysisController && silenceRoot.analysisController.busy
                from: 0
                to: 100
                value: silenceRoot.analysisController ? silenceRoot.analysisController.progress : 0
            }

            Item { Layout.fillHeight: true }
            }

            SilenceThresholdPicker {
                id: thresholdPicker
                initialValue: silenceThresholdSlider.value
                analysisController: silenceRoot.analysisController
                selectedTrackIndices: silenceTrackPanel.getSelectedIndices()
                timelineBackend: silenceRoot.appController ? silenceRoot.appController.timelineBackend : null
                onApplied: (value) => silenceThresholdSlider.value = value
            }
        }
    }

    Component {
        id: blackToolView
        ColumnLayout {
            id: blackRoot
            property var appController: null
            property bool mediaReady: false
            property var analysisController: null

            spacing: Theme.spacingSm
            Layout.fillWidth: true
            Layout.fillHeight: true

            // Toggle detection area overlay when this view is shown/hidden
            // Use onAnalysisControllerChanged because Component.onCompleted runs
            // before the Loader's onLoaded binds the property
            onAnalysisControllerChanged: {
                if (blackRoot.analysisController) {
                    blackRoot.analysisController.setShowDetectionOverlay(true)
                }
            }
            Component.onDestruction: {
                if (blackRoot.analysisController) {
                    blackRoot.analysisController.setShowDetectionOverlay(false)
                }
            }

            // DEFINE BLACK section
            Label {
                text: qsTr("DEFINE BLACK")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Avg. brightness under")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: avgBrightnessSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 255
                    stepSize: 1
                    value: 90
                    onValueChanged: if (blackRoot.analysisController) blackRoot.analysisController.clearBlackDetection()
                }

                Label {
                    text: qsTr("Max brightness under")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: maxBrightnessSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 255
                    stepSize: 1
                    value: 30
                    onValueChanged: if (blackRoot.analysisController) blackRoot.analysisController.clearBlackDetection()
                }

                Label {
                    text: qsTr("For at least")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: blackMinFramesSlider
                    Layout.fillWidth: true
                    from: 1
                    to: 100
                    stepSize: 1
                    value: 3
                    suffix: " frames"
                    onValueChanged: if (blackRoot.analysisController) blackRoot.analysisController.clearBlackDetection()
                }
            }

            Label {
                text: qsTr("Frame is black if the pixel brightness values are below thresholds")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // DETECTION AREA section
            Label {
                text: qsTr("DETECTION AREA")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            Label {
                text: qsTr("Black detection is only ran in the selected area. Drag handles on video preview to set detection area")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
            }

            Label {
                text: qsTr("T: %1%  L: %2%  R: %3%  B: %4%")
                    .arg(Math.round(blackRoot.analysisController ? blackRoot.analysisController.detectionAreaTop : 0))
                    .arg(Math.round(blackRoot.analysisController ? blackRoot.analysisController.detectionAreaLeft : 0))
                    .arg(Math.round(blackRoot.analysisController ? blackRoot.analysisController.detectionAreaRight : 0))
                    .arg(Math.round(blackRoot.analysisController ? blackRoot.analysisController.detectionAreaBottom : 0))
                font.pixelSize: 12
                font.family: "monospace"
                color: Theme.textSecondary
                Layout.fillWidth: true
            }

            AbstractButton {
                id: resetAreaButton
                text: qsTr("Reset to Full Frame")
                enabled: blackRoot.analysisController && (
                    blackRoot.analysisController.detectionAreaTop > 0 ||
                    blackRoot.analysisController.detectionAreaLeft > 0 ||
                    blackRoot.analysisController.detectionAreaRight > 0 ||
                    blackRoot.analysisController.detectionAreaBottom > 0
                )
                onClicked: {
                    if (blackRoot.analysisController) {
                        blackRoot.analysisController.resetDetectionArea()
                    }
                }
                Layout.fillWidth: true
                Layout.preferredHeight: 32
                Layout.bottomMargin: Theme.spacingMd

                background: Rectangle {
                    radius: Theme.panelRadius - 4
                    color: resetAreaButton.enabled
                        ? (resetAreaButton.hovered ? Theme.surfaceRaised : Theme.surface)
                        : Theme.surface
                    border.color: resetAreaButton.enabled
                        ? (resetAreaButton.hovered ? Theme.surfaceBorder : Theme.surfaceBorder)
                        : Qt.rgba(Theme.surfaceBorder.r, Theme.surfaceBorder.g, Theme.surfaceBorder.b, 0.5)
                    border.width: 1
                    opacity: resetAreaButton.enabled ? 1.0 : 0.5

                    Behavior on color { ColorAnimation { duration: 100 } }
                }

                contentItem: Label {
                    text: resetAreaButton.text
                    color: resetAreaButton.enabled ? Theme.textPrimary : Theme.textSecondary
                    font.pixelSize: 13
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            // SELECTION SCOPE section
            readonly property bool hasSelection: {
                let sel = blackRoot.appController?.timelineBackend?.selectionRange
                return sel && sel.length === 2 && sel[1] > sel[0]
            }

            CheckBox {
                id: limitToSelectionSwitch
                Layout.fillWidth: true
                Layout.topMargin: Theme.spacingSm
                text: qsTr("Limit black detection to selected area")
                checked: false
                onCheckedChanged: {
                    // Clear cache when toggling since results would be different
                    if (blackRoot.analysisController) {
                        blackRoot.analysisController.clearBlackDetection()
                    }
                }

                contentItem: Label {
                    text: limitToSelectionSwitch.text
                    font: limitToSelectionSwitch.font
                    color: Theme.textPrimary
                    verticalAlignment: Text.AlignVCenter
                    leftPadding: limitToSelectionSwitch.indicator.width + limitToSelectionSwitch.spacing
                    elide: Text.ElideRight
                }
            }

            Label {
                text: qsTr("Only detect black frames within the timeline selection")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // ACTIONS section
            Label {
                text: qsTr("ACTIONS")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            PrimaryButton {
                Layout.fillWidth: true
                text: {
                    if (blackRoot.analysisController && blackRoot.analysisController.busy) {
                        return qsTr("Detecting…")
                    }
                    if (blackRoot.analysisController && blackRoot.analysisController.blackVisualizationVisible) {
                        return qsTr("Hide Black/White Sections")
                    }
                    return qsTr("Show Black/White Sections")
                }
                enabled: blackRoot.mediaReady && blackRoot.analysisController && !blackRoot.analysisController.busy
                onClicked: {
                    if (!blackRoot.analysisController) {
                        return
                    }
                    // If already visible, hide
                    if (blackRoot.analysisController.blackVisualizationVisible) {
                        blackRoot.analysisController.setBlackVisualizationVisible(false)
                        return
                    }

                    // Determine time range based on selection switch
                    let rangeStart = -1.0
                    let rangeEnd = -1.0
                    if (limitToSelectionSwitch.checked && blackRoot.hasSelection) {
                        // Always invalidate cache when limited to selection,
                        // since the selection may have changed
                        blackRoot.analysisController.clearBlackDetection()
                        let sel = blackRoot.appController.timelineBackend.selectionRange
                        rangeStart = sel[0]
                        rangeEnd = sel[1]
                    }

                    // Show visualization (uses cache if available, runs detection if not)
                    blackRoot.analysisController.showBlackWhiteSections(
                        avgBrightnessSlider.value,
                        maxBrightnessSlider.value,
                        blackMinFramesSlider.value,
                        rangeStart,
                        rangeEnd
                    )
                }
            }

            Item { Layout.preferredHeight: Theme.spacingSm }

            Button {
                Layout.fillWidth: true
                text: blackRoot.analysisController && blackRoot.analysisController.busy
                      ? qsTr("Detecting…")
                      : qsTr("Cut Black Sections")
                enabled: blackRoot.mediaReady && blackRoot.analysisController && !blackRoot.analysisController.busy
                onClicked: {
                    if (blackRoot.analysisController) {
                        // Determine time range based on selection switch
                        let rangeStart = -1.0
                        let rangeEnd = -1.0
                        if (limitToSelectionSwitch.checked && blackRoot.hasSelection) {
                            // Always invalidate cache when limited to selection,
                            // since the selection may have changed
                            blackRoot.analysisController.clearBlackDetection()
                            let sel = blackRoot.appController.timelineBackend.selectionRange
                            rangeStart = sel[0]
                            rangeEnd = sel[1]
                        }

                        blackRoot.analysisController.runBlackDetectionAndCut(
                            avgBrightnessSlider.value,
                            maxBrightnessSlider.value,
                            blackMinFramesSlider.value,
                            blackClearExisting.checked,
                            rangeStart,
                            rangeEnd
                        )
                    }
                }
            }

            CheckBox {
                id: blackClearExisting
                Layout.fillWidth: true
                text: qsTr("Clear existing cuts before applying")
                checked: false
                enabled: blackRoot.mediaReady

                contentItem: Label {
                    text: blackClearExisting.text
                    font: blackClearExisting.font
                    color: Theme.textPrimary
                    opacity: blackClearExisting.enabled ? 1.0 : 0.5
                    verticalAlignment: Text.AlignVCenter
                    leftPadding: blackClearExisting.indicator.width + blackClearExisting.spacing
                    elide: Text.ElideRight
                }
            }

            ProgressBar {
                Layout.fillWidth: true
                visible: blackRoot.analysisController && blackRoot.analysisController.busy
                from: 0
                to: 100
                value: blackRoot.analysisController ? blackRoot.analysisController.progress : 0
            }

            Item { Layout.fillHeight: true }
        }
    }

    Component {
        id: speechlessToolView
        ColumnLayout {
            id: speechlessRoot
            property var appController: null
            property bool mediaReady: false
            property var analysisController: null

            spacing: Theme.spacingSm
            Layout.fillWidth: true
            Layout.fillHeight: true

            // DEFINE SPEECH section
            Label {
                text: qsTr("DEFINE SPEECH")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Confidence above")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: vadThresholdSlider
                    Layout.fillWidth: true
                    from: 10
                    to: 95
                    stepSize: 5
                    value: 50
                    suffix: "%"
                }

                Label {
                    text: qsTr("For more than")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: vadMinSpeechDurationSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 10
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Audio below threshold or shorter than minimum is treated as silence")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // DEFINE SILENCE section
            Label {
                text: qsTr("DEFINE SILENCE")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("No speech for")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: vadMinSilenceDurationSlider
                    Layout.fillWidth: true
                    from: 5
                    to: 100
                    stepSize: 1
                    value: 15
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Gaps shorter than this are not cut")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // PADDING section
            Label {
                text: qsTr("PADDING")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
            }

            GridLayout {
                columns: 2
                columnSpacing: Theme.spacingMd
                rowSpacing: Theme.spacingSm
                Layout.fillWidth: true

                Label {
                    text: qsTr("Pre-speech")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: vadPaddingBeforeSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 2
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }

                Label {
                    text: qsTr("Post-speech")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                }
                LabeledSlider {
                    id: vadPaddingAfterSlider
                    Layout.fillWidth: true
                    from: 0
                    to: 100
                    stepSize: 1
                    value: 2
                    decimals: 1
                    displayScale: 0.1
                    suffix: " s"
                }
            }

            Label {
                text: qsTr("Extra time kept around detected speech")
                font.pixelSize: 12
                color: Theme.textSecondary
                opacity: 0.7
                wrapMode: Text.WordWrap
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingMd
            }

            // TRACKS section - only show if multiple tracks
            Label {
                text: qsTr("TRACKS")
                font.pixelSize: 12
                font.weight: Font.DemiBold
                color: Theme.textSecondary
                opacity: 0.8
                visible: speechlessTrackPanel.shouldShow
            }

            Forms.TrackSelectionPanel {
                id: speechlessTrackPanel
                Layout.fillWidth: true
                visible: shouldShow
                tracks: speechlessRoot.analysisController ? speechlessRoot.analysisController.tracks : []
                settingsModel: root.settingsModel
                helperText: qsTr("Select which audio tracks to analyze for speech")
            }

            CheckBox {
                id: vadClearExisting
                objectName: "speechlessClearCheckbox"
                Layout.fillWidth: true
                text: qsTr("Clear existing cuts before applying")
                checked: false
                enabled: speechlessRoot.mediaReady

                contentItem: Label {
                    text: vadClearExisting.text
                    font: vadClearExisting.font
                    color: Theme.textPrimary
                    opacity: vadClearExisting.enabled ? 1.0 : 0.5
                    verticalAlignment: Text.AlignVCenter
                    leftPadding: vadClearExisting.indicator.width + vadClearExisting.spacing
                    elide: Text.ElideRight
                }

                onCheckedChanged: {
                    if (root.appController && root.appController.hintManager) {
                        root.appController.hintManager.dismissHint("speechless_clear_checkbox")
                    }
                }
            }

            PrimaryButton {
                objectName: "speechlessApplyButton"
                Layout.fillWidth: true
                text: speechlessRoot.analysisController && speechlessRoot.analysisController.busy
                      ? qsTr("Analyzing…")
                      : qsTr("Cut Speechless Regions")
                enabled: speechlessRoot.mediaReady && speechlessRoot.analysisController && !speechlessRoot.analysisController.busy
                onClicked: {
                    if (!speechlessRoot.analysisController) {
                        return
                    }
                    let selectedTracks = speechlessTrackPanel.getSelectedIndices()
                    speechlessRoot.analysisController.runSpeechlessDetection(
                        vadThresholdSlider.value / 100.0,
                        vadMinSpeechDurationSlider.value / 10.0,
                        vadMinSilenceDurationSlider.value / 10.0,
                        vadPaddingBeforeSlider.value / 10.0,
                        vadPaddingAfterSlider.value / 10.0,
                        vadClearExisting.checked,
                        selectedTracks
                    )
                    if (root.appController && root.appController.hintManager) {
                        root.appController.hintManager.dismissHint("speechless_apply")
                    }
                }
            }

            ProgressBar {
                Layout.fillWidth: true
                visible: speechlessRoot.analysisController && speechlessRoot.analysisController.busy
                from: 0
                to: 100
                value: speechlessRoot.analysisController ? speechlessRoot.analysisController.progress : 0
            }

            Item { Layout.fillHeight: true }
        }
    }
}
