pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import Qt5Compat.GraphicalEffects
import styles 1.0
import components.controls 1.0 as Controls
import components.layout 1.0

Frame {
    id: root
    required property var timelineBackend
    property var settingsModel: null
    property var exportFormController: null

    // Use audioTracksModel.tracks for sidebar (stable TrackState objects, no list rebuilds)
    readonly property var tracks: timelineBackend && timelineBackend.audioTracksModel
        ? timelineBackend.audioTracksModel.tracks : []
    readonly property bool mixEnabled: exportFormController ? exportFormController.mix_enabled : false
    property bool anySliderPressed: false
    property bool helpExpanded: false

    function indexForValue(value, options) {
        if (!options || value === undefined || value === null) return 0
        for (let i = 0; i < options.length; ++i) {
            if (options[i].value === value) return i
        }
        return 0
    }

    padding: 0

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surface
        border.color: Theme.surfaceBorder
    }

    contentItem: ColumnLayout {
        spacing: 0

        SidebarHeader {
            title: qsTr("Audio Mix")
        }

        ScrollView {
            id: scrollView
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true
            contentWidth: availableWidth

            // Disable Flickable scrolling while dragging volume sliders to prevent drag stealing
            Binding {
                target: scrollView.contentItem
                property: "interactive"
                value: !root.anySliderPressed
            }

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                // Enable mix toggle section
                ColumnLayout {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd

                    // Enable mix toggle
                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        Switch {
                            id: includeMixSwitch
                            checked: root.exportFormController ? root.exportFormController.mix_enabled : false
                            onToggled: {
                                if (root.exportFormController) {
                                    root.exportFormController.set_mix_enabled(checked)
                                }
                                // Reset track volumes when disabling mix
                                if (!checked && root.timelineBackend) {
                                    root.timelineBackend.resetAllTracks()
                                }
                            }

                            Connections {
                                target: root.exportFormController || null
                                ignoreUnknownSignals: true
                                function onForm_state_changed() {
                                    if (root.exportFormController && includeMixSwitch.checked !== root.exportFormController.mix_enabled) {
                                        includeMixSwitch.checked = root.exportFormController.mix_enabled
                                    }
                                }
                            }
                        }

                        Label {
                            Layout.fillWidth: true
                            text: qsTr("Enable audio mix")
                            color: Theme.textPrimary
                            font.pixelSize: 14
                            wrapMode: Text.WordWrap
                        }
                    }

                    // Collapsible "What's this?" section
                    Item {
                        Layout.fillWidth: true
                        implicitHeight: helpRow.implicitHeight

                        MouseArea {
                            anchors.fill: parent
                            cursorShape: Qt.PointingHandCursor
                            onClicked: root.helpExpanded = !root.helpExpanded
                        }

                        RowLayout {
                            id: helpRow
                            anchors.fill: parent
                            spacing: Theme.spacingXs

                            Image {
                                id: chevronIcon
                                Layout.preferredWidth: 16
                                Layout.preferredHeight: 16
                                source: root.helpExpanded ? Theme.iconUrl("chevron-up") : Theme.iconUrl("chevron-down")
                                visible: false
                            }

                            ColorOverlay {
                                Layout.preferredWidth: 16
                                Layout.preferredHeight: 16
                                source: chevronIcon
                                color: Theme.textSecondary
                                opacity: 0.6
                            }

                            Label {
                                Layout.fillWidth: true
                                text: qsTr("What's this?")
                                color: Theme.textSecondary
                                font.pixelSize: 13
                                opacity: 0.8
                            }
                        }
                    }

                    // Expandable help text
                    Label {
                        visible: root.helpExpanded
                        Layout.fillWidth: true
                        text: qsTr("When enabled, all audio tracks are combined into a single mixed track using your volume and mute settings below.")
                        color: Theme.textSecondary
                        font.pixelSize: 13
                        opacity: 0.7
                        wrapMode: Text.WordWrap
                    }
                }

                // Separator
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 1
                    color: Theme.surfaceBorder
                }

                // Volume controls section
                ColumnLayout {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd
                    opacity: root.mixEnabled ? 1.0 : 0.5

                    Label {
                        text: qsTr("Track Volumes")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    // Track volume sliders
                    Repeater {
                        model: root.tracks
                        delegate: ColumnLayout {
                            id: trackDelegate
                            required property var modelData  // TrackState QObject
                            required property int index
                            property var backend: root.timelineBackend  // qmllint disable unqualified

                            // Alias for cleaner access to the TrackState
                            readonly property var track: modelData

                            Layout.fillWidth: true
                            spacing: Theme.spacingXs

                            RowLayout {
                                Layout.fillWidth: true
                                spacing: Theme.spacingSm
                                visible: trackDelegate.track !== null

                                // Track color indicator
                                Rectangle {
                                    Layout.preferredWidth: 12
                                    Layout.preferredHeight: 12
                                    radius: 6
                                    color: trackDelegate.track && trackDelegate.track.color.valid ? trackDelegate.track.color : Theme.accent
                                }

                                Label {
                                    Layout.fillWidth: true
                                    text: trackDelegate.track ? trackDelegate.track.label : ""
                                    color: trackDelegate.track && trackDelegate.track.muted ? Theme.textSecondary : Theme.textPrimary
                                    font.pixelSize: 14
                                    elide: Text.ElideRight
                                }

                                // Mute button
                                RoundButton {
                                    Layout.preferredWidth: 32
                                    Layout.preferredHeight: 32
                                    padding: 0
                                    flat: true
                                    enabled: root.mixEnabled
                                    display: AbstractButton.IconOnly
                                    icon.source: trackDelegate.track && trackDelegate.track.muted ? Theme.iconUrl("volume-off") : Theme.iconUrl("volume")
                                    icon.width: 18
                                    icon.height: 18
                                    icon.color: trackDelegate.track && trackDelegate.track.muted ? Theme.textSecondary : Theme.textPrimary
                                    onClicked: {
                                        if (trackDelegate.backend && trackDelegate.track) {
                                            trackDelegate.backend.setTrackMuted(trackDelegate.track.index, !trackDelegate.track.muted)
                                        }
                                    }
                                    ToolTip.visible: hovered
                                    ToolTip.delay: 300
                                    ToolTip.text: trackDelegate.track && trackDelegate.track.muted ? qsTr("Unmute") : qsTr("Mute")
                                }
                            }

                            // Volume slider
                            RowLayout {
                                Layout.fillWidth: true
                                spacing: Theme.spacingSm
                                visible: trackDelegate.track !== null

                                Slider {
                                    id: volumeSlider
                                    Layout.fillWidth: true
                                    from: 0
                                    to: 200
                                    stepSize: 1
                                    enabled: root.mixEnabled && trackDelegate.track && !trackDelegate.track.muted

                                    // Track the model value separately - this binding always stays in sync
                                    readonly property real modelValue: trackDelegate.track ? trackDelegate.track.volume * 100 : 100

                                    // Sync slider to model when not dragging
                                    onModelValueChanged: {
                                        if (!pressed) {
                                            value = modelValue
                                        }
                                    }

                                    // Initialize on component creation
                                    Component.onCompleted: value = modelValue

                                    onPressedChanged: {
                                        root.anySliderPressed = pressed
                                        // Sync back to model value when releasing
                                        if (!pressed) {
                                            value = modelValue
                                        }
                                    }

                                    onMoved: {
                                        if (trackDelegate.backend && trackDelegate.track) {
                                            trackDelegate.backend.setTrackVolume(trackDelegate.track.index, value / 100.0)
                                        }
                                    }
                                }

                                Label {
                                    Layout.preferredWidth: 48
                                    text: Math.round(volumeSlider.value) + "%"
                                    color: trackDelegate.track && trackDelegate.track.muted ? Theme.textSecondary : Theme.textPrimary
                                    font.pixelSize: 13
                                    horizontalAlignment: Text.AlignRight
                                }
                            }
                        }
                    }

                    // Empty state
                    Label {
                        visible: root.tracks.length === 0
                        Layout.fillWidth: true
                        text: qsTr("No audio tracks available")
                        color: Theme.textSecondary
                        font.pixelSize: 13
                        wrapMode: Text.WordWrap
                    }
                }

                // Separator
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 1
                    color: Theme.surfaceBorder
                    Layout.topMargin: Theme.spacingMd
                }

                // Output Settings Section
                ColumnLayout {
                    id: outputSettingsSection
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    Layout.topMargin: Theme.spacingSm
                    spacing: Theme.spacingMd
                    opacity: root.mixEnabled ? 1.0 : 0.5

                    Label {
                        text: qsTr("Output Settings")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    // Auto/Manual toggle - segmented buttons
                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        Controls.SegmentedButton {
                            id: autoMixBtn
                            Layout.fillWidth: true
                            text: qsTr("Auto")
                            checked: !root.timelineBackend || root.timelineBackend.mixMode !== "manual"
                            enabled: root.mixEnabled
                            onClicked: root.timelineBackend && root.timelineBackend.setMixMode("auto")
                        }

                        Controls.SegmentedButton {
                            id: manualMixBtn
                            Layout.fillWidth: true
                            text: qsTr("Manual")
                            checked: root.timelineBackend && root.timelineBackend.mixMode === "manual"
                            enabled: root.mixEnabled
                            onClicked: root.timelineBackend && root.timelineBackend.setMixMode("manual")
                        }
                    }

                    // Manual settings (only visible when manual mode selected)
                    GridLayout {
                        Layout.fillWidth: true
                        columns: 2
                        columnSpacing: Theme.spacingMd
                        rowSpacing: Theme.spacingMd
                        visible: manualMixBtn.checked

                        Label {
                            text: qsTr("Codec")
                            color: Theme.textSecondary
                            font.pixelSize: 12
                        }
                        ComboBox {
                            Layout.fillWidth: true
                            enabled: root.mixEnabled
                            model: root.timelineBackend ? root.timelineBackend.validAudioCodecs : []
                            textRole: "label"
                            valueRole: "value"
                            currentIndex: root.indexForValue(
                                root.timelineBackend ? root.timelineBackend.mixCodec : "",
                                root.timelineBackend ? root.timelineBackend.validAudioCodecs : []
                            )
                            onActivated: index => {
                                if (root.timelineBackend && model[index]) {
                                    root.timelineBackend.setMixCodec(model[index].value)
                                }
                            }
                        }

                        Label {
                            text: qsTr("Sample Rate")
                            color: Theme.textSecondary
                            font.pixelSize: 12
                        }
                        ComboBox {
                            Layout.fillWidth: true
                            enabled: root.mixEnabled
                            model: root.timelineBackend ? root.timelineBackend.sampleRateOptions : []
                            textRole: "label"
                            valueRole: "value"
                            currentIndex: root.indexForValue(
                                root.timelineBackend ? root.timelineBackend.mixSampleRate : 0,
                                root.timelineBackend ? root.timelineBackend.sampleRateOptions : []
                            )
                            onActivated: index => {
                                if (root.timelineBackend && model[index]) {
                                    root.timelineBackend.setMixSampleRate(model[index].value)
                                }
                            }
                        }

                        Label {
                            text: qsTr("Channels")
                            color: Theme.textSecondary
                            font.pixelSize: 12
                        }
                        ComboBox {
                            Layout.fillWidth: true
                            enabled: root.mixEnabled
                            model: root.timelineBackend ? root.timelineBackend.channelOptions : []
                            textRole: "label"
                            valueRole: "value"
                            currentIndex: root.indexForValue(
                                root.timelineBackend ? root.timelineBackend.mixChannels : "",
                                root.timelineBackend ? root.timelineBackend.channelOptions : []
                            )
                            onActivated: index => {
                                if (root.timelineBackend && model[index]) {
                                    root.timelineBackend.setMixChannels(model[index].value)
                                }
                            }
                        }

                        Label {
                            text: qsTr("Bitrate")
                            color: Theme.textSecondary
                            font.pixelSize: 12
                        }
                        ComboBox {
                            Layout.fillWidth: true
                            enabled: root.mixEnabled
                            model: root.timelineBackend ? root.timelineBackend.bitrateOptions : []
                            textRole: "label"
                            valueRole: "value"
                            currentIndex: root.indexForValue(
                                root.timelineBackend ? root.timelineBackend.mixBitrate : 0,
                                root.timelineBackend ? root.timelineBackend.bitrateOptions : []
                            )
                            onActivated: index => {
                                if (root.timelineBackend && model[index]) {
                                    root.timelineBackend.setMixBitrate(model[index].value)
                                }
                            }
                        }
                    }

                    // Auto mode explanation
                    Label {
                        Layout.fillWidth: true
                        text: qsTr("Auto mode selects optimal codec based on output format.")
                        font.pixelSize: 11
                        color: Theme.textSecondary
                        opacity: 0.7
                        wrapMode: Text.WordWrap
                        visible: autoMixBtn.checked
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }
}
