import QtQuick 2.15
import QtQuick.Layouts 1.15
import components.layout 1.0
import components.media 1.0
import components.workspace 1.0
import features.cut 1.0
import features.analysis 1.0
import styles 1.0

Item {
    id: root

    required property var appWorkspaceController
    required property var settingsModel

    property string demoSource: Qt.resolvedUrl("../../../../test_input/sneezing_goat.mp4")
    property bool __syncing: false

    onVisibleChanged: {
        if (!visible) {
            analysisSidebar.selectedTool = ""
        }
    }

    readonly property var appController: root.appWorkspaceController
    readonly property var workspaceController: appController.cutWorkspaceController
    readonly property var timelineBackend: appController.timelineBackend
    readonly property real mediaDuration: appController.mediaDuration
    readonly property bool playbackActive: appController.playing
    readonly property string playbackStatusText: appController.mediaStatus === "ready"
                                                    ? appController.mediaDisplayName || qsTr("Media loaded")
                                                    : appController.mediaStatus === "loading"
                                                          ? qsTr("Loading media…")
                                                          : appController.mediaStatus === "error"
                                                                ? appController.mediaError || qsTr("Media error")
                                                                : qsTr("No media loaded")
    readonly property bool timelineReady: appController.mediaStatus === "ready"
    readonly property string timelineEmptyText: appController.mediaStatus === "loading"
                                                      ? qsTr("Building timeline…")
                                                      : appController.mediaStatus === "error"
                                                            ? appController.mediaError || qsTr("Timeline unavailable")
                                                            : qsTr("Open a media file to begin")

    function seekToTime(seconds) {
        const time = Math.max(0, Number(seconds) || 0)
        if (root.__syncing) {
            return
        }
        root.__syncing = true
        if (appController) {
            appController.seekToTime(time)
        } else if (workspaceController && workspaceController.seekToTime) {
            workspaceController.seekToTime(time)
        }
        root.__syncing = false
    }

    WorkspaceViewPreferences {
        id: viewPrefs
        settingsModel: root.settingsModel
        workspaceKey: "cut"
    }

    Connections {
        target: root.appController
        function onRequestFitToView() {
            timelineSection.fitToDuration()
        }
    }

    ColumnLayout {
        anchors.fill: parent
        spacing: Theme.spacingLg

        RowLayout {
            Layout.fillWidth: true
            Layout.fillHeight: true
            spacing: Theme.spacingLg

            ToolSidebar {
                id: analysisSidebar
                visible: viewPrefs.prefs.showToolsSidebar
                Layout.preferredWidth: 320
                Layout.fillHeight: true
                appController: root.appController
                settingsModel: root.settingsModel
            }

            ColumnLayout {
                Layout.fillWidth: true
                Layout.fillHeight: true
                spacing: Theme.spacingLg

                MediaPreviewCard {
                    appController: root.appController
                    workspaceController: root.workspaceController
                    showFrameStrip: viewPrefs.prefs.showFrameStrip
                }

                // Timecode display
                Rectangle {
                    id: timecodeRect
                    Layout.fillWidth: true
                    Layout.preferredHeight: 28
                    color: Theme.surface
                    radius: 4
                    visible: viewPrefs.prefs.showPlayheadInfo

                    property real currentTime: root.timelineBackend ? root.timelineBackend.playheadTime : 0
                    property bool hasVideo: root.appController ? root.appController.hasVideo : false
                    property int currentFrame: {
                        if (!hasVideo || !root.timelineReady || currentTime < 0) return 0
                        return root.appController.frameIndexAtTime(currentTime)
                    }
                    property int totalFrames: {
                        if (!hasVideo || !root.timelineReady) return 0
                        return root.appController.frameIndexAtTime(root.mediaDuration)
                    }

                    function formatTime(t) {
                        var hrs = Math.floor(t / 3600)
                        var mins = Math.floor((t % 3600) / 60)
                        var secs = Math.floor(t % 60)
                        var ms = Math.floor((t % 1) * 1000)
                        var hh = hrs.toString().padStart(2, '0')
                        var mm = mins.toString().padStart(2, '0')
                        var ss = secs.toString().padStart(2, '0')
                        var mmm = ms.toString().padStart(3, '0')
                        return hh + ":" + mm + ":" + ss + "." + mmm
                    }

                    function parseTime(str) {
                        // Parse HH:MM:SS.mmm or MM:SS.mmm or SS.mmm or just seconds
                        var parts = str.trim().split(":")
                        var seconds = 0
                        if (parts.length === 3) {
                            seconds = parseFloat(parts[0]) * 3600 + parseFloat(parts[1]) * 60 + parseFloat(parts[2])
                        } else if (parts.length === 2) {
                            seconds = parseFloat(parts[0]) * 60 + parseFloat(parts[1])
                        } else {
                            seconds = parseFloat(parts[0])
                        }
                        return isNaN(seconds) ? -1 : seconds
                    }

                    Row {
                        anchors.centerIn: parent
                        spacing: Theme.spacingLg

                        // Time input
                        Rectangle {
                            width: timeInput.visible ? timeInput.width : timeLabel.width + 8
                            height: 22
                            radius: 3
                            color: timeMouseArea.containsMouse ? Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.15) : "transparent"

                            Text {
                                id: timeLabel
                                anchors.centerIn: parent
                                visible: !timeInput.visible
                                color: Theme.textSecondary
                                font.family: "monospace"
                                font.pixelSize: 13
                                text: timecodeRect.formatTime(timecodeRect.currentTime) + " / " + timecodeRect.formatTime(root.mediaDuration)
                            }

                            MouseArea {
                                id: timeMouseArea
                                anchors.fill: parent
                                hoverEnabled: true
                                cursorShape: Qt.PointingHandCursor
                                visible: !timeInput.visible
                                onClicked: {
                                    timeInput.text = timeLabel.text
                                    timeInput.visible = true
                                    timeInput.selectAll()
                                    timeInput.forceActiveFocus()
                                }
                            }

                            TextInput {
                                id: timeInput
                                anchors.centerIn: parent
                                visible: false
                                font.family: "monospace"
                                font.pixelSize: 13
                                color: Theme.textPrimary
                                selectionColor: Theme.accent
                                selectedTextColor: "#ffffff"
                                horizontalAlignment: Text.AlignHCenter

                                function commitAndClose() {
                                    var t = timecodeRect.parseTime(text)
                                    if (t >= 0 && root.appController) {
                                        root.seekToTime(Math.max(0, Math.min(t, root.mediaDuration)))
                                    }
                                    close()
                                }

                                function close() {
                                    visible = false
                                    focus = false
                                    root.forceActiveFocus()
                                }

                                onAccepted: commitAndClose()
                                onActiveFocusChanged: {
                                    if (!activeFocus) visible = false
                                }
                                Keys.onEscapePressed: close()
                            }
                        }

                        Rectangle {
                            visible: timecodeRect.hasVideo
                            width: 1
                            height: 16
                            anchors.verticalCenter: parent.verticalCenter
                            color: Theme.surfaceBorder
                        }

                        // Frame input (only for video files)
                        Rectangle {
                            visible: timecodeRect.hasVideo
                            width: frameInput.visible ? frameInput.width + 16 : frameLabel.width + 8
                            height: 22
                            radius: 3
                            color: frameMouseArea.containsMouse ? Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.15) : "transparent"

                            Text {
                                id: frameLabel
                                anchors.centerIn: parent
                                visible: !frameInput.visible
                                color: Theme.textSecondary
                                font.family: "monospace"
                                font.pixelSize: 13
                                text: qsTr("Frame %1 / %2").arg(timecodeRect.currentFrame).arg(timecodeRect.totalFrames)
                            }

                            MouseArea {
                                id: frameMouseArea
                                anchors.fill: parent
                                hoverEnabled: true
                                cursorShape: Qt.PointingHandCursor
                                visible: !frameInput.visible
                                onClicked: {
                                    frameInput.text = timecodeRect.currentFrame.toString()
                                    frameInput.visible = true
                                    frameInput.selectAll()
                                    frameInput.forceActiveFocus()
                                }
                            }

                            TextInput {
                                id: frameInput
                                anchors.centerIn: parent
                                visible: false
                                font.family: "monospace"
                                font.pixelSize: 13
                                color: Theme.textPrimary
                                selectionColor: Theme.accent
                                selectedTextColor: "#ffffff"
                                horizontalAlignment: Text.AlignHCenter
                                validator: IntValidator { bottom: 0 }

                                function commitAndClose() {
                                    var frame = parseInt(text)
                                    if (!isNaN(frame) && root.appController) {
                                        var t = root.appController.frameTimeForIndex(frame)
                                        if (t >= 0) {
                                            root.seekToTime(Math.min(t, root.mediaDuration))
                                        }
                                    }
                                    close()
                                }

                                function close() {
                                    visible = false
                                    focus = false
                                    root.forceActiveFocus()
                                }

                                onAccepted: commitAndClose()
                                onActiveFocusChanged: {
                                    if (!activeFocus) visible = false
                                }
                                Keys.onEscapePressed: close()
                            }
                        }
                    }
                }

                TimelineSection {
                    id: timelineSection
                    Layout.fillWidth: true
                    timelineBackend: root.timelineBackend
                    appController: root.appController
                    showOverviewTimeline: viewPrefs.prefs.showOverviewTimeline
                    showAudioTimeline: viewPrefs.prefs.showAudioTimeline
                    timelineReady: root.timelineReady
                    emptyStateText: root.timelineEmptyText
                }

                // Cut actions toolbar
                Card {
                    Layout.preferredHeight: 40
                    Layout.topMargin: -Theme.spacingSm
                    contentPadding: Theme.spacingXs
                    visible: viewPrefs.prefs.showCutActionsToolbar

                    CutActionsBar {
                        anchors.fill: parent
                        workspaceController: root.workspaceController
                        timelineBackend: root.timelineBackend
                        appWorkspaceController: root.appController
                    }
                }
            }

            // File list panel (shows joined files)
            FileListPanel {
                id: fileListPanel
                visible: viewPrefs.prefs.showFileListPanel && root.appController.joinModel && root.appController.joinModel.fileCount > 1
                Layout.preferredWidth: 280
                Layout.fillHeight: true
                appController: root.appController
                joinModel: root.appController ? root.appController.joinModel : null
            }
        }
    }
}
