pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Dialogs

import styles 1.0
import components.layout 1.0

Frame {
    id: root

    required property var appController
    property var joinModel: null

    readonly property int fileCount: joinModel && typeof joinModel.fileCount !== 'undefined' ? joinModel.fileCount : 0
    readonly property real totalDuration: joinModel && typeof joinModel.totalDuration !== 'undefined' ? Number(joinModel.totalDuration) : 0.0

    padding: 0

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surface
        border.color: Theme.surfaceBorder
    }

    FileDialog {
        id: fileDialog
        title: qsTr("Add media files")
        nameFilters: ["Video files (*.mp4 *.mkv *.mov *.avi *.webm *.m4v)", "All files (*)"]
        fileMode: FileDialog.OpenFiles
        onAccepted: {
            for (var i = 0; i < selectedFiles.length; i++) {
                var path = selectedFiles[i].toString()
                if (path.startsWith("file://")) {
                    path = path.substring(7)
                }
                root.appController.addMedia(path)
            }
        }
    }

    contentItem: ColumnLayout {
        spacing: 0

        SidebarHeader {
            title: qsTr("Files")
        }

        ScrollView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingMd

                // File count and total duration
                ColumnLayout {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingMd
                    spacing: Theme.spacingSm

                    RowLayout {
                        Layout.fillWidth: true

                        Label {
                            text: qsTr("Joined Files")
                            font.weight: Font.DemiBold
                            color: Theme.textPrimary
                            Layout.fillWidth: true
                        }

                        Button {
                            id: addButton
                            text: qsTr("Add")
                            onClicked: fileDialog.open()

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: addButton.hovered ? Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.2) : "transparent"
                                border.width: 1
                                border.color: Theme.accent
                            }

                            contentItem: Label {
                                text: addButton.text
                                color: Theme.accent
                                horizontalAlignment: Text.AlignHCenter
                            }
                        }
                    }

                    Label {
                        visible: root.fileCount > 0
                        text: qsTr("%1 file(s), %2 total").arg(root.fileCount).arg(formatDuration(root.totalDuration))
                        font.pixelSize: 12
                        color: Theme.textSecondary

                        function formatDuration(seconds) {
                            var mins = Math.floor(seconds / 60)
                            var secs = Math.floor(seconds % 60)
                            return mins + ":" + (secs < 10 ? "0" : "") + secs
                        }
                    }
                }

                // File list
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: Math.min(200, Math.max(60, fileListView.contentHeight + 16))
                    Layout.leftMargin: Theme.spacingMd
                    Layout.rightMargin: Theme.spacingMd
                    color: Theme.background
                    radius: Theme.panelRadius - 4
                    border.width: 1
                    border.color: Theme.surfaceBorder

                    ListView {
                        id: fileListView
                        anchors.fill: parent
                        anchors.margins: 8
                        model: root.joinModel && root.joinModel.files ? root.joinModel.files : []
                        spacing: 4
                        clip: true

                        delegate: Rectangle {
                            id: fileDelegate
                            width: fileListView.width
                            height: 36
                            radius: 4
                            color: mouseArea.containsMouse ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.05) : "transparent"

                            required property var modelData
                            required property int index

                            MouseArea {
                                id: mouseArea
                                anchors.fill: parent
                                hoverEnabled: true
                            }

                            RowLayout {
                                anchors.fill: parent
                                anchors.leftMargin: 8
                                anchors.rightMargin: 8
                                spacing: 8

                                // Index number
                                Label {
                                    text: (fileDelegate.index + 1) + "."
                                    color: Theme.textSecondary
                                    font.pixelSize: 12
                                    Layout.preferredWidth: 20
                                }

                                // File name
                                Label {
                                    text: fileDelegate.modelData.displayName
                                    color: Theme.textPrimary
                                    elide: Text.ElideMiddle
                                    Layout.fillWidth: true
                                }

                                // Duration
                                Label {
                                    text: formatDuration(fileDelegate.modelData.duration)
                                    color: Theme.textSecondary
                                    font.pixelSize: 12

                                    function formatDuration(seconds) {
                                        var mins = Math.floor(seconds / 60)
                                        var secs = Math.floor(seconds % 60)
                                        return mins + ":" + (secs < 10 ? "0" : "") + secs
                                    }
                                }

                                // Move up button
                                Button {
                                    id: moveUpBtn
                                    visible: fileDelegate.index > 0
                                    implicitWidth: 20
                                    implicitHeight: 20
                                    flat: true
                                    onClicked: root.appController.reorderMedia(fileDelegate.index, fileDelegate.index - 1)

                                    contentItem: Label {
                                        text: "\u25B2"
                                        font.pixelSize: 10
                                        color: Theme.textSecondary
                                        horizontalAlignment: Text.AlignHCenter
                                        verticalAlignment: Text.AlignVCenter
                                    }

                                    background: Rectangle {
                                        radius: 4
                                        color: moveUpBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.1) : "transparent"
                                    }
                                }

                                // Move down button
                                Button {
                                    id: moveDownBtn
                                    visible: fileDelegate.index < root.fileCount - 1
                                    implicitWidth: 20
                                    implicitHeight: 20
                                    flat: true
                                    onClicked: root.appController.reorderMedia(fileDelegate.index, fileDelegate.index + 1)

                                    contentItem: Label {
                                        text: "\u25BC"
                                        font.pixelSize: 10
                                        color: Theme.textSecondary
                                        horizontalAlignment: Text.AlignHCenter
                                        verticalAlignment: Text.AlignVCenter
                                    }

                                    background: Rectangle {
                                        radius: 4
                                        color: moveDownBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.1) : "transparent"
                                    }
                                }

                                // Remove button (only show if more than 1 file)
                                Button {
                                    id: removeBtn
                                    visible: root.fileCount > 1
                                    implicitWidth: 20
                                    implicitHeight: 20
                                    flat: true
                                    onClicked: root.appController.removeMedia(fileDelegate.index)

                                    contentItem: Label {
                                        text: "\u00D7"
                                        font.pixelSize: 14
                                        color: Theme.textSecondary
                                        horizontalAlignment: Text.AlignHCenter
                                        verticalAlignment: Text.AlignVCenter
                                    }

                                    background: Rectangle {
                                        radius: 4
                                        color: removeBtn.hovered ? Qt.rgba(255, 0, 0, 0.1) : "transparent"
                                    }
                                }
                            }
                        }

                        // Empty state
                        Label {
                            visible: fileListView.count === 0
                            anchors.centerIn: parent
                            text: qsTr("No files loaded")
                            color: Theme.textSecondary
                        }
                    }
                }

                // Spacer
                Item {
                    Layout.fillHeight: true
                }
            }
        }
    }
}
