import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Dialogs

import styles 1.0
import components.controls 1.0 as Controls
import components.forms 1.0 as FormComponents
import "../../../utils" as Utils

pragma ComponentBehavior: Bound

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    signal requestNavigateToAudioMix()

    // qmllint disable missing-property
    property var appWorkspaceController
    // qmllint enable missing-property
    property var workspaceController: appWorkspaceController
    property var exportController: workspaceController ? workspaceController.exportController : null
    property var exportFormController: workspaceController ? workspaceController.exportFormController : null
    property var executionModel: workspaceController ? workspaceController.exportExecutionModel : null
    // qmllint disable missing-property
    property var settingsModel
    // qmllint enable missing-property

    property int currentTabIndex: 0

    function loadTabIndex() {
        if (root.settingsModel) {
            var prefs = root.settingsModel.export_preferences();
            root.currentTabIndex = prefs.hasOwnProperty("export_tab_index") ? prefs["export_tab_index"] : 0;
        }
    }

    function saveTabIndex(idx) {
        if (root.settingsModel) {
            root.settingsModel.set_export_preference("export_tab_index", idx);
        }
    }

    readonly property var exportTabs: [
        { key: "media", label: qsTr("Media Export") },
        { key: "timeline", label: qsTr("Timeline Export") }
    ]

    property var currentForm: ({})
    property var validationState: ({})
    property var trackLabels: []
    property var presetOptions: []
    property string activePresetName: ""
    property bool isApplyingPreset: false
    property bool encodeSettingsExpanded: false
    property int videoQualityIndex: 0
    property int xmlModeIndex: 0
    readonly property string statusName: executionModel ? executionModel.statusValue : "idle"
    property int progressCurrent: 0
    property int progressTotal: 0
    property string exportMessage: ""
    property bool showProgress: statusName === "running"

    width: parent ? Math.min(parent.width * 0.85, 1080) : 900
    height: parent ? Math.min(parent.height * 0.85, 740) : 640
    anchors.centerIn: parent

    background: Rectangle {
        anchors.fill: parent
        radius: Theme.panelRadius
        color: Theme.surfaceStrong
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    function indexForValue(value, options) {
        if (!options || value === undefined || value === null) {
            return 0;
        }
        for (let i = 0; i < options.length; ++i) {
            if (options[i].value === value) {
                return i;
            }
        }
        return 0;
    }

    function formatTime(t) {
        var hrs = Math.floor(t / 3600)
        var mins = Math.floor((t % 3600) / 60)
        var secs = Math.floor(t % 60)
        var ms = Math.floor((t % 1) * 1000)
        var hh = hrs.toString().padStart(2, '0')
        var mm = mins.toString().padStart(2, '0')
        var ss = secs.toString().padStart(2, '0')
        var mmm = ms.toString().padStart(3, '0')
        return hh + ":" + mm + ":" + ss + "." + mmm
    }

    function syncState() {
        if (!root.workspaceController) {
            root.currentForm = ({});
            root.validationState = ({ is_valid: false, error: qsTr("No project loaded"), warnings: [] });
            root.trackLabels = [];
            root.videoQualityIndex = 0;
            root.xmlModeIndex = 0;
            return;
        }
        if (!root.exportFormController) {
            return;
        }
        // Build form state snapshot from exportFormController properties
        root.currentForm = {
            output_path: root.exportFormController.output_path || "",
            segment_mode: root.exportFormController.segment_mode || false,
            selected_tracks: root.exportFormController.selected_tracks || [],
            mix_enabled: root.exportFormController.mix_enabled || false,
            video_mode: root.exportFormController.video_mode || 0,
            video_quality_index: root.exportFormController.video_quality || 0,
            video_encoder_override: root.exportFormController.video_encoder_override || "",
            has_video: root.workspaceController ? root.workspaceController.hasVideo : false,
            xml_output_path: root.exportFormController.xml_path || "",
            xml_timeline_name: root.exportFormController.xml_timeline_name || "",
            xml_mode: root.exportFormController.xml_mode || 0,
            xml_path_with_extension: root.exportFormController.xml_path_with_extension || ""
        };
        root.validationState = {
            is_valid: root.exportFormController.is_valid || false,
            error: root.exportFormController.validation_error || "",
            error_code: root.exportFormController.validation_error_code || "",
            warnings: root.exportFormController.validation_warnings || []
        };
        root.trackLabels = root.exportFormController.track_labels || [];
        root.presetOptions = root.exportFormController.preset_options || [];
        const qualityOptions = root.exportFormController.video_quality_options || [];
        const xmlOptions = root.exportFormController.xml_mode_options || [];
        root.videoQualityIndex = root.indexForValue(root.currentForm.video_quality_index, qualityOptions);
        root.xmlModeIndex = root.indexForValue(root.currentForm.xml_mode, xmlOptions);
    }

    Component.onCompleted: syncState()
    onVisibleChanged: {
        if (root.appWorkspaceController && root.appWorkspaceController.hintManager) {
            root.appWorkspaceController.hintManager.set_export_overlay_open(visible)
        }
        if (visible) {
            syncState();
            loadTabIndex();
        }
    }

    contentItem: ColumnLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingXl
        spacing: 0 // Spacing handled by children

        // --- Header ---
        RowLayout {
            Layout.fillWidth: true
            Layout.bottomMargin: Theme.spacingLg
            spacing: Theme.spacingMd

            Label {
                text: qsTr("Export")
                font.pixelSize: 24
                font.bold: true
                color: Theme.textPrimary
                Layout.fillWidth: true
            }

            // Preset Menu Button
            Controls.SecondaryButton {
                text: qsTr("Presets")
                visible: root.presetOptions && root.presetOptions.length > 0
                onClicked: presetMenu.open()

                Menu {
                    id: presetMenu
                    y: parent.height

                    Repeater {
                        model: root.presetOptions ? root.presetOptions.length : 0
                        MenuItem {
                            required property int index
                            property var option: root.presetOptions[index]
                            text: option ? option.label : ""
                            onTriggered: {
                                if (!option) return;
                                root.isApplyingPreset = true;
                                root.activePresetName = option.value;
                                if (root.exportFormController) {
                                    root.exportFormController.apply_preset(option.value);
                                }
                                root.isApplyingPreset = false;
                            }
                        }
                    }

                    MenuSeparator { visible: root.presetOptions && root.presetOptions.length > 0 }

                    MenuItem {
                        text: qsTr("Save current settings as preset...")
                        onTriggered: {
                            presetNameInput.text = root.activePresetName || "";
                            presetDialog.open();
                        }
                    }

                    MenuItem {
                        text: qsTr("Reset to defaults")
                        onTriggered: root.exportFormController && root.exportFormController.reset_to_defaults()
                    }
                }
            }

            ToolButton {
                implicitWidth: 36
                implicitHeight: 36
                text: "\u2715"
                font.pixelSize: 18
                onClicked: root.close()
                Accessible.name: qsTr("Close export panel")
            }
        }

        // --- Tab Bar ---
        RowLayout {
            Layout.fillWidth: true
            Layout.bottomMargin: Theme.spacingMd
            spacing: Theme.spacingMd

            Repeater {
                model: root.exportTabs.length

                delegate: AbstractButton {
                    id: tabBtn
                    required property int index
                    property var tabEntry: root.exportTabs[index]
                    objectName: tabEntry ? tabEntry.key + "ExportTab" : ""

                    text: tabEntry ? tabEntry.label : ""
                    checked: root.currentTabIndex === index
                    onClicked: {
                        // Dismiss timeline export tab hint when clicking on timeline tab
                        if (tabEntry && tabEntry.key === "timeline" && root.appWorkspaceController && root.appWorkspaceController.hintManager) {
                            root.appWorkspaceController.hintManager.dismissHint("timeline_export_tab")
                        }
                        root.currentTabIndex = index;
                        root.saveTabIndex(index);
                    }

                    hoverEnabled: true
                    font.pixelSize: 13
                    font.weight: checked ? Font.DemiBold : Font.Normal
                    implicitHeight: 36
                    leftPadding: Theme.spacingMd
                    rightPadding: Theme.spacingMd

                    contentItem: Label {
                        text: tabBtn.text
                        font: tabBtn.font
                        color: tabBtn.checked ? Theme.textPrimary : (tabBtn.hovered ? Theme.textPrimary : Theme.textSecondary)
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        Behavior on color { ColorAnimation { duration: 150 } }
                    }

                    background: Rectangle {
                        color: tabBtn.hovered && !tabBtn.checked ? Theme.surfaceRaised : "transparent"
                        radius: 4

                        Rectangle {
                            anchors.bottom: parent.bottom
                            anchors.horizontalCenter: parent.horizontalCenter
                            width: tabBtn.checked ? parent.width - 16 : 0
                            height: 3
                            radius: 1.5
                            color: Theme.accent
                            Behavior on width { NumberAnimation { duration: 200; easing.type: Easing.OutQuad } }
                        }
                    }
                }
            }

            Item { Layout.fillWidth: true }
        }

        // --- Main Content ---
        RowLayout {
            Layout.fillWidth: true
            Layout.fillHeight: true
            spacing: Theme.spacingXl

            // Left Column: Tab Content
            StackLayout {
                Layout.fillWidth: true
                Layout.fillHeight: true
                currentIndex: root.currentTabIndex

                // Tab 0: Media Export
                ScrollView {
                    clip: true
                    contentWidth: availableWidth

                    ColumnLayout {
                        width: parent.width
                        spacing: Theme.spacingLg

                        // Output Section
                        Rectangle {
                            Layout.fillWidth: true
                            color: Theme.surface
                            radius: Theme.panelRadius
                            border.color: Theme.surfaceBorder
                            border.width: 1
                            implicitHeight: outputCol.implicitHeight + Theme.spacingLg * 2

                            ColumnLayout {
                                id: outputCol
                                anchors.top: parent.top
                                anchors.left: parent.left
                                anchors.right: parent.right
                                anchors.margins: Theme.spacingLg
                                spacing: Theme.spacingMd

                                Label {
                                    text: qsTr("Output")
                                    font.weight: Font.DemiBold
                                    color: Theme.textPrimary
                                }

                                FormComponents.LabeledField {
                                    label: qsTr("File path")

                                    RowLayout {
                                        Layout.fillWidth: true
                                        spacing: Theme.spacingSm

                                        TextField {
                                            objectName: "exportOutputPathField"
                                            Layout.fillWidth: true
                                            placeholderText: Qt.platform.os === "windows" ? "C:\\Videos\\output.mp4" : "/path/to/output.mp4"
                                            text: root.currentForm.output_path || ""
                                            selectByMouse: true
                                            onTextEdited: root.exportFormController && root.exportFormController.set_output_path(text)
                                            onActiveFocusChanged: {
                                                if (activeFocus && root.appWorkspaceController && root.appWorkspaceController.hintManager) {
                                                    root.appWorkspaceController.hintManager.dismissHint("export_output_path")
                                                }
                                            }
                                        }

                                        Button {
                                            text: qsTr("Browse")
                                            onClicked: outputPathDialog.openWithCurrentPath()
                                        }
                                    }
                                }

                                RowLayout {
                                    spacing: Theme.spacingSm
                                    Switch {
                                        checked: !!root.currentForm.segment_mode
                                        onToggled: root.exportFormController && root.exportFormController.set_segment_mode(checked)
                                    }
                                    Label {
                                        text: qsTr("Export each individual segment to its own file")
                                        color: Theme.textSecondary
                                    }
                                }
                            }
                        }

                        // Video & Audio Section
                        Rectangle {
                            Layout.fillWidth: true
                            color: Theme.surface
                            radius: Theme.panelRadius
                            border.color: Theme.surfaceBorder
                            border.width: 1
                            implicitHeight: mediaCol.implicitHeight + Theme.spacingLg * 2

                            ColumnLayout {
                                id: mediaCol
                                anchors.top: parent.top
                                anchors.left: parent.left
                                anchors.right: parent.right
                                anchors.margins: Theme.spacingLg
                                spacing: Theme.spacingMd

                                Label {
                                    text: qsTr("Audio tracks")
                                    font.weight: Font.DemiBold
                                    color: Theme.textPrimary
                                }

                                // Audio mix toggle
                                RowLayout {
                                    Layout.fillWidth: true
                                    spacing: Theme.spacingSm

                                    Switch {
                                        id: mixToggle
                                        checked: !!root.currentForm.mix_enabled
                                        onToggled: root.exportFormController && root.exportFormController.set_mix_enabled(checked)
                                    }

                                    Label {
                                        text: qsTr("Enable audio mix")
                                        color: Theme.textPrimary
                                        font.pixelSize: 13
                                    }

                                    Item { Layout.fillWidth: true }

                                    Label {
                                        visible: mixToggle.checked
                                        text: qsTr("Configure audio mix...")
                                        color: Theme.accent
                                        font.pixelSize: 12

                                        MouseArea {
                                            anchors.fill: parent
                                            cursorShape: Qt.PointingHandCursor
                                            onClicked: {
                                                root.close()
                                                root.requestNavigateToAudioMix()
                                            }
                                        }
                                    }
                                }

                                // Audio Tracks
                                Label {
                                    visible: !!root.currentForm.mix_enabled
                                    text: qsTr("Passthru Audio Tracks")
                                    font.pixelSize: 13
                                    color: Theme.textSecondary
                                }

                                Repeater {
                                    model: root.trackLabels
                                    delegate: CheckDelegate {
                                        required property var modelData
                                        required property int index
                                        Layout.fillWidth: true
                                        text: modelData.label
                                        checked: root.currentForm.selected_tracks ? !!root.currentForm.selected_tracks[index] : false
                                        onToggled: {
                                            if (!root.workspaceController) return;
                                            const selection = root.currentForm.selected_tracks ? root.currentForm.selected_tracks.slice() : [];
                                            selection[index] = checked;
                                            if (root.exportFormController) {
                                                root.exportFormController.set_selected_tracks(selection);
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        // Video Settings Card
                        Rectangle {
                            visible: !!root.currentForm.has_video
                            Layout.fillWidth: true
                            color: Theme.surface
                            radius: Theme.panelRadius
                            border.color: Theme.surfaceBorder
                            border.width: 1
                            implicitHeight: videoCol.implicitHeight + Theme.spacingLg * 2

                            ColumnLayout {
                                id: videoCol
                                anchors.top: parent.top
                                anchors.left: parent.left
                                anchors.right: parent.right
                                anchors.margins: Theme.spacingLg
                                spacing: Theme.spacingMd

                                Label {
                                    text: qsTr("Video")
                                    font.weight: Font.DemiBold
                                    color: Theme.textPrimary
                                }

                                // Export Mode - segmented buttons
                                Label {
                                    text: qsTr("Export Mode")
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                }

                                RowLayout {
                                    Layout.fillWidth: true
                                    spacing: Theme.spacingSm

                                    Controls.SegmentedButton {
                                        id: smartCutBtn
                                        Layout.fillWidth: true
                                        text: qsTr("Smart Cut")
                                        checked: root.currentForm.video_mode === 1
                                        onClicked: root.exportFormController && root.exportFormController.set_video_mode(1)
                                    }

                                    Controls.SegmentedButton {
                                        id: fullRecodeBtn
                                        Layout.fillWidth: true
                                        text: qsTr("Full Recode")
                                        checked: root.currentForm.video_mode === 3
                                        onClicked: root.exportFormController && root.exportFormController.set_video_mode(3)
                                    }
                                }

                                // Info text about current mode
                                Label {
                                    Layout.fillWidth: true
                                    text: smartCutBtn.checked
                                        ? qsTr("Preserves original quality, only recodes around cut points.")
                                        : qsTr("Re-encodes the entire video. Slower but allows codec changes.")
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                    wrapMode: Text.WordWrap
                                }

                                // Collapsible Encode Settings
                                Item {
                                    Layout.fillWidth: true
                                    Layout.topMargin: Theme.spacingSm
                                    implicitHeight: encodeSettingsHeader.implicitHeight

                                    RowLayout {
                                        id: encodeSettingsHeader
                                        anchors.fill: parent
                                        spacing: Theme.spacingSm

                                        Label {
                                            text: root.encodeSettingsExpanded ? "▼" : "▶"
                                            font.pixelSize: 10
                                            color: Theme.textSecondary
                                        }

                                        Label {
                                            text: qsTr("Encode settings")
                                            font.pixelSize: 12
                                            color: Theme.textSecondary
                                            Layout.fillWidth: true
                                        }
                                    }

                                    MouseArea {
                                        anchors.fill: parent
                                        cursorShape: Qt.PointingHandCursor
                                        onClicked: root.encodeSettingsExpanded = !root.encodeSettingsExpanded
                                    }
                                }

                                // Encode settings content (collapsible)
                                ColumnLayout {
                                    visible: root.encodeSettingsExpanded
                                    Layout.fillWidth: true
                                    Layout.leftMargin: Theme.spacingMd
                                    spacing: Theme.spacingMd

                                    // Codec dropdown - only for Full Recode
                                    Label {
                                        text: qsTr("Codec")
                                        font.pixelSize: 12
                                        color: Theme.textSecondary
                                        visible: fullRecodeBtn.checked
                                    }

                                    ComboBox {
                                        Layout.fillWidth: true
                                        visible: fullRecodeBtn.checked
                                        model: root.exportFormController ? root.exportFormController.valid_video_codecs : []
                                        textRole: "label"
                                        valueRole: "value"
                                        currentIndex: {
                                            // Find index matching current video_encoder_override
                                            if (!root.exportFormController) return 0
                                            const codec = root.exportFormController.video_encoder_override || ""
                                            const opts = root.exportFormController.valid_video_codecs || []
                                            for (let i = 0; i < opts.length; i++) {
                                                if (opts[i].value === codec) return i
                                            }
                                            return 0
                                        }
                                        onActivated: index => {
                                            if (root.exportFormController && model[index]) {
                                                root.exportFormController.set_video_encoder_override(model[index].value)
                                            }
                                        }
                                    }

                                    // Quality dropdown
                                    Label {
                                        text: qsTr("Quality")
                                        font.pixelSize: 12
                                        color: Theme.textSecondary
                                    }

                                    ComboBox {
                                        Layout.fillWidth: true
                                        model: root.exportFormController ? root.exportFormController.video_quality_options : []
                                        textRole: "label"
                                        valueRole: "value"
                                        currentIndex: model && model.length > 0 ? Math.min(root.videoQualityIndex, model.length - 1) : 0
                                        onActivated: index => root.exportFormController && root.exportFormController.set_video_quality(model[index].value)
                                    }

                                    // Quality explanation
                                    Label {
                                        Layout.fillWidth: true
                                        text: smartCutBtn.checked
                                            ? qsTr("Affects quality of recoded portions around cut points.")
                                            : qsTr("Affects quality of the entire video.")
                                        font.pixelSize: 11
                                        color: Theme.textSecondary
                                        opacity: 0.7
                                        wrapMode: Text.WordWrap
                                    }
                                }
                            }
                        }
                    }
                }

                // Tab 1: Timeline Export
                ScrollView {
                    clip: true
                    contentWidth: availableWidth

                    ColumnLayout {
                        width: parent.width
                        spacing: Theme.spacingLg

                        Rectangle {
                            Layout.fillWidth: true
                            color: Theme.surface
                            radius: Theme.panelRadius
                            border.color: Theme.surfaceBorder
                            border.width: 1
                            implicitHeight: timelineCol.implicitHeight + Theme.spacingLg * 2

                            ColumnLayout {
                                id: timelineCol
                                anchors.top: parent.top
                                anchors.left: parent.left
                                anchors.right: parent.right
                                anchors.margins: Theme.spacingLg
                                spacing: Theme.spacingMd

                                Label {
                                    text: qsTr("Timeline Settings")
                                    font.weight: Font.DemiBold
                                    color: Theme.textPrimary
                                }

                                GridLayout {
                                    columns: 2
                                    columnSpacing: Theme.spacingLg
                                    rowSpacing: Theme.spacingMd

                                    Label { text: qsTr("Format"); color: Theme.textSecondary }
                                    ComboBox {
                                        Layout.fillWidth: true
                                        model: root.exportFormController ? root.exportFormController.xml_mode_options : []
                                        textRole: "label"
                                        valueRole: "value"
                                        currentIndex: model && model.length > 0 ? Math.min(root.xmlModeIndex, model.length - 1) : 0
                                        onActivated: index => root.exportFormController && root.exportFormController.set_xml_mode(model[index].value)
                                    }

                                    Label { text: qsTr("Sequence Name"); color: Theme.textSecondary }
                                    TextField {
                                        Layout.fillWidth: true
                                        placeholderText: qsTr("My Edit")
                                        text: root.currentForm.xml_timeline_name || ""
                                        onTextEdited: root.exportFormController && root.exportFormController.set_xml_timeline_name(text)
                                    }

                                    Label { text: qsTr("Path"); color: Theme.textSecondary }
                                    RowLayout {
                                        Layout.fillWidth: true
                                        spacing: Theme.spacingSm
                                        TextField {
                                            Layout.fillWidth: true
                                            placeholderText: Qt.platform.os === "windows" ? "C:\\Documents\\sequence.otio" : "/path/to/sequence.otio"
                                            text: root.currentForm.xml_path_with_extension || ""
                                            selectByMouse: true
                                            onTextEdited: root.exportFormController && root.exportFormController.set_xml_path(text)
                                        }
                                        Button {
                                            text: qsTr("Browse")
                                            onClicked: xmlPathDialog.openWithCurrentPath()
                                        }
                                    }
                                }

                                RowLayout {
                                    Layout.topMargin: Theme.spacingSm
                                    spacing: Theme.spacingSm
                                    Switch {
                                        checked: root.exportFormController ? root.exportFormController.xml_ripple_mode : true
                                        onToggled: root.exportFormController && root.exportFormController.set_xml_ripple_mode(checked)
                                    }
                                    Label {
                                        text: qsTr("Ripple edit (remove gaps between segments)")
                                        color: Theme.textSecondary
                                    }
                                }

                                Label {
                                    Layout.topMargin: Theme.spacingMd
                                    text: qsTr("Export your edit decisions as an interchange file for use in video editing software like DaVinci Resolve, Premiere Pro, or Final Cut Pro.")
                                    wrapMode: Text.WordWrap
                                    color: Theme.textSecondary
                                    font.pixelSize: 12
                                    Layout.fillWidth: true
                                }
                            }
                        }
                    }
                }
            }

            // Right Column: Status & Validation (Fixed Width)
            Rectangle {
                Layout.preferredWidth: 300
                Layout.fillHeight: true
                color: Theme.surface
                radius: Theme.panelRadius
                border.color: Theme.surfaceBorder
                border.width: 1

                ColumnLayout {
                    anchors.fill: parent
                    anchors.margins: Theme.spacingLg
                    spacing: Theme.spacingMd

                    Label {
                        text: qsTr("Status")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    // Status Indicator
                    RowLayout {
                        spacing: Theme.spacingSm
                        Rectangle {
                            Layout.preferredWidth: 12
                            Layout.preferredHeight: 12
                            radius: 6
                            color: root.statusName === "running" ? Theme.accent :
                                   root.statusName === "completed" ? "#4CAF50" :
                                   root.statusName === "error" ? "#F44336" : Theme.textSecondary
                        }
                        Label {
                            text: root.statusName === "running" ? qsTr("Exporting...") :
                                  root.statusName === "completed" ? qsTr("Finished") :
                                  root.statusName === "error" ? qsTr("Error") :
                                  root.statusName === "cancelled" ? qsTr("Cancelled") : qsTr("Ready")
                            font.bold: true
                        }
                    }

                    // Output Length
                    RowLayout {
                        spacing: Theme.spacingSm
                        Label {
                            text: qsTr("Output length:")
                            color: Theme.textSecondary
                        }
                        Label {
                            text: root.exportFormController ? root.formatTime(root.exportFormController.export_duration) : "00:00:00.000"
                            color: Theme.textPrimary
                        }
                    }

                    ProgressBar {
                        Layout.fillWidth: true
                        visible: root.showProgress
                        from: 0
                        to: root.progressTotal > 0 ? root.progressTotal : 1
                        value: root.progressTotal > 0 ? root.progressCurrent : 0
                        indeterminate: root.progressTotal <= 0
                    }

                    Label {
                        visible: root.showProgress && root.progressTotal > 0
                        text: qsTr("%1 / %2").arg(root.progressCurrent).arg(root.progressTotal)
                        color: Theme.textSecondary
                        font.pixelSize: 12
                    }

                    // Messages
                    ScrollView {
                        id: messagesScrollView
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        clip: true

                        ColumnLayout {
                            width: messagesScrollView.availableWidth
                            spacing: Theme.spacingSm

                            Label {
                                visible: root.exportMessage.length > 0
                                text: root.exportMessage
                                wrapMode: Text.WordWrap
                                color: Theme.textPrimary
                                Layout.fillWidth: true
                            }

                            Label {
                                visible: root.statusName === "error" && root.executionModel !== null && (root.executionModel.error_message ?? "") !== ""
                                text: (root.executionModel && root.executionModel.error_message) ? root.executionModel.error_message : ""
                                wrapMode: Text.WordWrap
                                color: Theme.accent
                                Layout.fillWidth: true
                            }

                            // Validation Errors/Warnings
                            Label {
                                visible: root.validationState && !root.validationState.is_valid && root.validationState.error
                                text: (root.validationState && root.validationState.error) ? root.validationState.error : ""
                                color: Theme.accent
                                font.bold: true
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }

                            Controls.SecondaryButton {
                                visible: root.validationState && !root.validationState.is_valid
                                         && root.validationState.error_code === "passthrough_codec_incompatible"
                                text: qsTr("Enable Audio Mix")
                                Layout.fillWidth: true
                                onClicked: {
                                    if (root.exportFormController) {
                                        root.exportFormController.set_mix_enabled(true)
                                        // Disable all passthru tracks
                                        const emptySelection = root.trackLabels.map(() => false);
                                        root.exportFormController.set_selected_tracks(emptySelection);
                                    }
                                }
                            }

                            Repeater {
                                model: root.validationState && root.validationState.warnings ? root.validationState.warnings : []
                                delegate: Label {
                                    required property string modelData
                                    text: "• " + modelData
                                    color: "#FF9800" // Warning color
                                    wrapMode: Text.WordWrap
                                    Layout.fillWidth: true
                                }
                            }
                        }
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Cancel Export")
                        Layout.fillWidth: true
                        visible: root.statusName === "running"
                        onClicked: root.workspaceController && root.workspaceController.cancelExport()
                    }
                }
            }
        }

        // --- Footer ---
        Rectangle {
            Layout.fillWidth: true
            Layout.preferredHeight: 1
            color: Theme.surfaceBorder
            Layout.topMargin: Theme.spacingMd
            Layout.bottomMargin: Theme.spacingMd
        }

        RowLayout {
            Layout.fillWidth: true
            spacing: Theme.spacingMd
            Layout.alignment: Qt.AlignRight

            Item { Layout.fillWidth: true } // Spacer

            Controls.SecondaryButton {
                text: qsTr("Close")
                onClicked: root.close()
            }

            Controls.PrimaryButton {
                objectName: "startExportButton"
                visible: root.currentTabIndex === 0
                text: qsTr("Start Export")
                enabled: root.workspaceController ? (!!root.validationState && !!root.validationState.is_valid && root.statusName !== "running") : false
                onClicked: {
                    if (root.appWorkspaceController && root.appWorkspaceController.hintManager) {
                        root.appWorkspaceController.hintManager.dismissHint("start_export")
                    }
                    if (root.workspaceController && root.workspaceController.beginExport()) {
                        root.exportMessage = "";
                    }
                }
            }

            Controls.PrimaryButton {
                visible: root.currentTabIndex === 1
                text: qsTr("Export Timeline")
                enabled: root.workspaceController ? (root.statusName !== "running") : false
                onClicked: root.workspaceController && root.workspaceController.exportTimelineXml()
            }
        }
    }

    Connections {
        target: root.exportFormController || null
        enabled: !!root.exportFormController
        ignoreUnknownSignals: true

        function onForm_state_changed() {
            if (!root.isApplyingPreset) {
                root.activePresetName = "";
            }
            root.syncState();
        }
        function onValidation_changed() { root.syncState() }
        function onTrack_labels_changed() { root.syncState() }
        function onPreset_options_changed() { root.syncState() }
    }

    Connections {
        target: root.executionModel || null
        enabled: !!root.executionModel
        ignoreUnknownSignals: true

        function onStatus_changed(status) {
            const effective = root.executionModel ? root.executionModel.statusValue : String(status || "idle");
            if (effective !== "running") {
                root.progressCurrent = 0;
                root.progressTotal = 0;
            } else {
                root.exportMessage = "";
            }
        }

        function onProgress_total_changed(value) {
            root.progressTotal = value;
            root.progressCurrent = 0;
        }

        function onProgress_changed(value) {
            root.progressCurrent = value;
        }

        function onExport_completed(message) {
            root.exportMessage = message || qsTr("Export complete");
        }

        function onExport_failed(message) {
            root.exportMessage = message || qsTr("Export failed");
        }

        function onExport_cancelled() {
            root.exportMessage = qsTr("Export cancelled");
        }
    }

    Connections {
        target: root.exportController || null
        enabled: !!root.exportController
        ignoreUnknownSignals: true

        function onXml_export_completed(message) {
            root.exportMessage = message || qsTr("Timeline export complete");
        }

        function onXml_export_failed(message) {
            root.exportMessage = message || qsTr("Timeline export failed");
        }
    }

    Dialog {
        id: presetDialog
        modal: true
        title: qsTr("Save export preset")
        standardButtons: Dialog.Ok | Dialog.Cancel
        onOpened: {
            presetNameInput.selectAll();
            presetNameInput.forceActiveFocus();
        }
        onAccepted: {
            if (!root.workspaceController) {
                return;
            }
            const name = presetNameInput.text.trim();
            if (!name) {
                return;
            }
            root.isApplyingPreset = true;
            const saved = root.exportFormController && root.exportFormController.save_preset(name);
            if (saved) {
                root.activePresetName = name;
            }
            root.isApplyingPreset = false;
            root.syncState();
        }

        contentItem: ColumnLayout {
            spacing: Theme.spacingSm
            width: 360

            Label {
                text: qsTr("Enter a name for this preset. Saving again with the same name will update it.")
                wrapMode: Text.WordWrap
                color: Theme.textSecondary
            }

            TextField {
                id: presetNameInput
                Layout.fillWidth: true
                placeholderText: qsTr("Preset name")
                selectByMouse: true
            }
        }
    }

    FileDialog {
        id: outputPathDialog
        title: qsTr("Export destination")
        fileMode: FileDialog.SaveFile
        nameFilters: [
            qsTr("Video Files") + " (*.mp4 *.mkv *.mov)",
            qsTr("Audio Files") + " (*.mp3 *.wav *.flac *.aac)",
            qsTr("All Files") + " (*)"
        ]

        function openWithCurrentPath() {
            const path = root.currentForm.output_path;
            if (path) {
                currentFolder = Utils.PathUtils.pathToFolderUrl(path);
                selectedFile = Utils.PathUtils.pathToUrl(path);
            }
            open();
        }

        onAccepted: {
            const filePath = Utils.PathUtils.urlToPath(selectedFile);
            if (filePath && root.exportFormController) {
                root.exportFormController.set_output_path(filePath);
            }
        }
    }

    FileDialog {
        id: xmlPathDialog
        title: qsTr("Timeline export path")
        fileMode: FileDialog.SaveFile
        nameFilters: [
            "DaVinci OTIO (*.otio)",
            "Final Cut Pro 7 XML (*.xml)",
            "Final Cut Pro X (*.fcpxml)",
            "Avid AAF (*.aaf)",
            "Shotcut MLT (*.mlt)",
            qsTr("All Files") + " (*)"
        ]

        function openWithCurrentPath() {
            const path = root.currentForm.xml_path_with_extension;
            if (path) {
                currentFolder = Utils.PathUtils.pathToFolderUrl(path);
                selectedFile = Utils.PathUtils.pathToUrl(path);
            }
            open();
        }

        onAccepted: {
            const filePath = Utils.PathUtils.urlToPath(selectedFile);
            if (filePath && root.exportFormController) {
                root.exportFormController.set_xml_path(filePath);
            }
        }
    }

}
