pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Dialogs
import Qt5Compat.GraphicalEffects

import styles 1.0
import components.layout 1.0

Frame {
    id: root

    // AppWorkspaceController for file management (uses shared join model)
    required property var appController

    // Convenience aliases
    readonly property var joinModel: appController ? appController.joinModel : null
    readonly property int fileCount: joinModel ? joinModel.fileCount : 0
    readonly property real totalDuration: joinModel ? Number(joinModel.totalDuration) : 0
    readonly property int heroIndex: joinModel ? joinModel.heroIndex : 0

    padding: 0

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surface
        border.color: Theme.surfaceBorder
    }

    FileDialog {
        id: fileDialog
        title: qsTr("Select media files to join")
        nameFilters: ["Video files (*.mp4 *.mkv *.mov *.avi *.webm *.m4v)", "All files (*)"]
        fileMode: FileDialog.OpenFiles
        onAccepted: {
            for (var i = 0; i < selectedFiles.length; i++) {
                var path = selectedFiles[i].toString()
                // Remove file:// prefix
                if (path.startsWith("file://")) {
                    path = path.substring(7)
                }
                root.appController.addMedia(path)
            }
        }
    }

    contentItem: ColumnLayout {
        spacing: 0

        SidebarHeader {
            title: qsTr("Join Files")
        }

        // Info section explaining lossless joining
        Rectangle {
            Layout.fillWidth: true
            Layout.margins: Theme.spacingMd
            Layout.topMargin: 0
            color: Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.1)
            radius: Theme.panelRadius - 4
            implicitHeight: infoContent.implicitHeight + Theme.spacingMd * 2

            ColumnLayout {
                id: infoContent
                anchors.fill: parent
                anchors.margins: Theme.spacingMd
                spacing: Theme.spacingXs

                Label {
                    Layout.fillWidth: true
                    text: qsTr("Lossless video joining")
                    font.weight: Font.DemiBold
                    font.pixelSize: 12
                    color: Theme.textPrimary
                }

                Label {
                    Layout.fillWidth: true
                    text: qsTr("Combine multiple video files without re-encoding. Compatible files are joined instantly with zero quality loss.")
                    wrapMode: Text.WordWrap
                    font.pixelSize: 11
                    color: Theme.textSecondary
                    lineHeight: 1.3
                }
            }
        }

        ScrollView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                // Files Section
                ColumnLayout {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd

                    RowLayout {
                        Layout.fillWidth: true

                        Label {
                            text: qsTr("Files")
                            font.weight: Font.DemiBold
                            color: Theme.textPrimary
                            Layout.fillWidth: true
                        }

                        Button {
                            id: addFilesButton
                            text: qsTr("Add")
                            onClicked: fileDialog.open()

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: addFilesButton.hovered ? Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.2) : "transparent"
                                border.width: 1
                                border.color: Theme.accent
                            }

                            contentItem: Label {
                                text: addFilesButton.text
                                color: Theme.accent
                                horizontalAlignment: Text.AlignHCenter
                            }
                        }
                    }

                    // File count and duration
                    Label {
                        visible: root.fileCount > 0
                        text: qsTr("%1 files, %2 total").arg(root.fileCount).arg(formatDuration(root.totalDuration))
                        font.pixelSize: 12
                        color: Theme.textSecondary

                        function formatDuration(seconds) {
                            var mins = Math.floor(seconds / 60)
                            var secs = Math.floor(seconds % 60)
                            return mins + ":" + (secs < 10 ? "0" : "") + secs
                        }
                    }

                    // File list
                    Rectangle {
                        Layout.fillWidth: true
                        Layout.preferredHeight: Math.min(300, Math.max(100, fileListView.contentHeight + 16))
                        color: Theme.background
                        radius: Theme.panelRadius - 4
                        border.width: 1
                        border.color: Theme.surfaceBorder

                        ListView {
                            id: fileListView
                            anchors.fill: parent
                            anchors.margins: 8
                            model: root.joinModel ? root.joinModel.files : []
                            spacing: 4
                            clip: true

                            delegate: Rectangle {
                                id: fileDelegate
                                width: fileListView.width
                                height: delegateContent.implicitHeight + 8
                                radius: 4

                                required property var modelData
                                required property int index

                                readonly property bool isHero: fileDelegate.index === root.heroIndex
                                readonly property var compat: fileDelegate.modelData.compatibility
                                readonly property string compatLevel: compat ? compat.level : "green"
                                readonly property var warnings: compat && compat.warnings ? compat.warnings : []

                                // Background with compatibility tint (only for incompatible files)
                                color: {
                                    if (mouseArea.containsMouse) {
                                        return Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.05)
                                    }
                                    switch (fileDelegate.compatLevel) {
                                        case "yellow": return Qt.rgba(1, 0.8, 0, 0.12)
                                        case "red": return Qt.rgba(1, 0, 0, 0.12)
                                        default: return "transparent"
                                    }
                                }

                                MouseArea {
                                    id: mouseArea
                                    anchors.fill: parent
                                    hoverEnabled: true
                                }

                                ColumnLayout {
                                    id: delegateContent
                                    anchors.left: parent.left
                                    anchors.right: parent.right
                                    anchors.top: parent.top
                                    anchors.margins: 4
                                    spacing: 2

                                    RowLayout {
                                        Layout.fillWidth: true
                                        spacing: 6

                                        // Star icon (clickable to set hero) + compatibility dot
                                        Row {
                                            spacing: 3

                                            Item {
                                                width: 16
                                                height: 16

                                                Image {
                                                    id: starIcon
                                                    anchors.fill: parent
                                                    source: fileDelegate.isHero ? "../../../../icons/star-filled.svg" : "../../../../icons/star.svg"
                                                    sourceSize.width: 16
                                                    sourceSize.height: 16
                                                    visible: false  // Hidden, ColorOverlay renders it
                                                }

                                                ColorOverlay {
                                                    anchors.fill: starIcon
                                                    source: starIcon
                                                    color: fileDelegate.isHero ? "#ffb800" : Theme.textSecondary
                                                }

                                                MouseArea {
                                                    id: starMouseArea
                                                    anchors.fill: parent
                                                    hoverEnabled: true
                                                    cursorShape: fileDelegate.isHero ? Qt.ArrowCursor : Qt.PointingHandCursor
                                                    onClicked: {
                                                        if (!fileDelegate.isHero) {
                                                            root.appController.setHeroFile(fileDelegate.index)
                                                        }
                                                    }
                                                }

                                                ToolTip {
                                                    visible: starMouseArea.containsMouse
                                                    text: fileDelegate.isHero ? qsTr("Main file - others are compared to this") : qsTr("Click to make this the main file")
                                                    delay: 500
                                                }
                                            }

                                            // Compatibility status dot
                                            Rectangle {
                                                id: compatDot
                                                width: 8
                                                height: 8
                                                radius: 4
                                                anchors.verticalCenter: parent.verticalCenter
                                                color: {
                                                    switch (fileDelegate.compatLevel) {
                                                        case "hero": return "#44bb44"  // Green for hero
                                                        case "green": return "#44bb44"
                                                        case "yellow": return "#ddaa00"
                                                        case "red": return "#dd4444"
                                                        default: return "#44bb44"
                                                    }
                                                }

                                                MouseArea {
                                                    id: dotMouseArea
                                                    anchors.fill: parent
                                                    anchors.margins: -4  // Larger hit area
                                                    hoverEnabled: true
                                                }

                                                ToolTip {
                                                    visible: dotMouseArea.containsMouse
                                                    text: {
                                                        switch (fileDelegate.compatLevel) {
                                                            case "hero": return qsTr("Compatible")
                                                            case "green": return qsTr("Compatible - no conversion needed")
                                                            case "yellow": return qsTr("Audio will be converted")
                                                            case "red": return qsTr("Video will be converted (slow)")
                                                            default: return ""
                                                        }
                                                    }
                                                    delay: 500
                                                }
                                            }
                                        }

                                        // Index number
                                        Label {
                                            text: (fileDelegate.index + 1) + "."
                                            color: Theme.textSecondary
                                            font.pixelSize: 12
                                            Layout.preferredWidth: 20
                                        }

                                        // File name
                                        Label {
                                            text: fileDelegate.modelData.displayName
                                            color: Theme.textPrimary
                                            elide: Text.ElideMiddle
                                            Layout.fillWidth: true
                                        }

                                        // Duration
                                        Label {
                                            text: formatDuration(fileDelegate.modelData.duration)
                                            color: Theme.textSecondary
                                            font.pixelSize: 12

                                            function formatDuration(seconds) {
                                                var mins = Math.floor(seconds / 60)
                                                var secs = Math.floor(seconds % 60)
                                                return mins + ":" + (secs < 10 ? "0" : "") + secs
                                            }
                                        }

                                        // Move up button
                                        Button {
                                            id: moveUpBtn
                                            visible: fileDelegate.index > 0
                                            implicitWidth: 24
                                            implicitHeight: 24
                                            flat: true
                                            onClicked: root.appController.reorderMedia(fileDelegate.index, fileDelegate.index - 1)

                                            contentItem: Label {
                                                text: "\u25B2"
                                                color: Theme.textSecondary
                                                horizontalAlignment: Text.AlignHCenter
                                                verticalAlignment: Text.AlignVCenter
                                            }

                                            background: Rectangle {
                                                radius: 4
                                                color: moveUpBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.1) : "transparent"
                                            }
                                        }

                                        // Move down button
                                        Button {
                                            id: moveDownBtn
                                            visible: fileDelegate.index < root.fileCount - 1
                                            implicitWidth: 24
                                            implicitHeight: 24
                                            flat: true
                                            onClicked: root.appController.reorderMedia(fileDelegate.index, fileDelegate.index + 1)

                                            contentItem: Label {
                                                text: "\u25BC"
                                                color: Theme.textSecondary
                                                horizontalAlignment: Text.AlignHCenter
                                                verticalAlignment: Text.AlignVCenter
                                            }

                                            background: Rectangle {
                                                radius: 4
                                                color: moveDownBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.1) : "transparent"
                                            }
                                        }

                                        // Remove button
                                        Button {
                                            id: removeBtn
                                            implicitWidth: 24
                                            implicitHeight: 24
                                            flat: true
                                            onClicked: root.appController.removeMedia(fileDelegate.index)

                                            contentItem: Label {
                                                text: "\u00D7"
                                                font.pixelSize: 16
                                                color: Theme.textSecondary
                                                horizontalAlignment: Text.AlignHCenter
                                                verticalAlignment: Text.AlignVCenter
                                            }

                                            background: Rectangle {
                                                radius: 4
                                                color: removeBtn.hovered ? Qt.rgba(255, 0, 0, 0.1) : "transparent"
                                            }
                                        }
                                    }

                                    // Compatibility warnings
                                    Column {
                                        Layout.fillWidth: true
                                        Layout.leftMargin: 22  // Align with filename (after star)
                                        visible: fileDelegate.warnings.length > 0
                                        spacing: 1

                                        Repeater {
                                            model: fileDelegate.warnings

                                            Label {
                                                required property var modelData
                                                text: modelData.text
                                                font.pixelSize: 10
                                                color: modelData.level === "red" ? "#ff4444" : "#cc9900"
                                            }
                                        }
                                    }
                                }
                            }

                            // Empty state
                            ColumnLayout {
                                visible: fileListView.count === 0
                                anchors.centerIn: parent
                                spacing: Theme.spacingSm

                                Label {
                                    Layout.alignment: Qt.AlignHCenter
                                    text: qsTr("No files added")
                                    color: Theme.textSecondary
                                }

                                Label {
                                    Layout.alignment: Qt.AlignHCenter
                                    text: qsTr("Click 'Add' to select files")
                                    font.pixelSize: 11
                                    color: Theme.textSecondary
                                    opacity: 0.7
                                }
                            }
                        }
                    }

                    // Tip about compatibility
                    Label {
                        Layout.fillWidth: true
                        visible: root.fileCount > 1
                        text: qsTr("Green = instant join. Yellow/red = will convert (slower).")
                        wrapMode: Text.WordWrap
                        font.pixelSize: 11
                        font.italic: true
                        color: Theme.textSecondary
                        opacity: 0.7
                    }
                }

                // Spacer
                Item {
                    Layout.fillHeight: true
                }
            }
        }
    }
}
