pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0
import SMC.Components 1.0

ColumnLayout {
    id: root
    objectName: "frameStrip"
    spacing: Theme.spacingXs

    required property var appWorkspaceController
    readonly property var appController: root.appWorkspaceController
    readonly property var frameCache: appController.frameCache
    readonly property var playbackController: appController.playbackController
    property int beforeCount: 4
    property int afterCount: 4

    Loader {
        Layout.fillWidth: true
        // Only build items when visible to avoid background churn
        active: root.visible

        property var appController: root.appController
        property var frameCache: root.frameCache
        property var playbackController: root.playbackController
        property int beforeCount: root.beforeCount
        property int afterCount: root.afterCount

        onLoaded: {
            item.appController = Qt.binding(() => appController)
            item.frameCache = Qt.binding(() => frameCache)
            item.playbackController = Qt.binding(() => playbackController)
            item.beforeCount = Qt.binding(() => beforeCount)
            item.afterCount = Qt.binding(() => afterCount)
        }

        sourceComponent: Component {
            RowLayout {
                id: stripRow
                Layout.fillWidth: true
                Layout.alignment: Qt.AlignHCenter
                spacing: Theme.spacingSm

                property var appController
                property var frameCache
                property var playbackController
                property int beforeCount
                property int afterCount

                // Cut and selection data triggers for property updates
                property var timelineBackend: appController ? appController.timelineBackend : null
                readonly property var cutRanges: timelineBackend ? timelineBackend.cutRanges : []
                readonly property var selectionRange: timelineBackend ? timelineBackend.selectionRange : []

                // Central frame index for bindings; stable items update their index
                readonly property int centerIndex: (appController && appController.mediaStatus === "ready" && appController.hasVideo)
                                                    ? appController.frameIndexAtTime(appController.playbackPosition)
                                                    : -1

                // Total frame count for range checks
                readonly property int mediaFrameCount: (appController && appController.mediaStatus === "ready" && appController.hasVideo)
                                                       ? appController.frameCount()
                                                       : 0

                // Dynamic frame sizing: maintain video aspect ratio, cap height for vertical videos
                readonly property int frameCount: beforeCount + 1 + afterCount
                readonly property real totalSpacing: (frameCount - 1) * spacing
                readonly property real videoAspectRatio: appController ? appController.videoAspectRatio : (16.0 / 9.0)
                // Base width if space divided equally - used as max height cap
                readonly property real baseFrameWidth: (width - totalSpacing) / frameCount
                // Height: natural height capped at baseFrameWidth (prevents tall vertical frames)
                readonly property real frameHeight: Math.min(baseFrameWidth / videoAspectRatio, baseFrameWidth)
                // Width: derived from height to maintain aspect ratio (no stretching)
                readonly property real frameWidth: frameHeight * videoAspectRatio

                // Keep a constant number of delegates to avoid create/destroy churn
                Repeater {
                    id: frameRepeater
                    model: stripRow.beforeCount + 1 + stripRow.afterCount
                    delegate: Rectangle {
                        id: frameDelegate
                        required property int index

                        // Compute frame index from stable offset relative to center
                        readonly property int offset: index - stripRow.beforeCount
                        // Use a distinct name to avoid clashing with VideoTextureItem.frameIndex
                        readonly property int computedFrameIndex: stripRow.centerIndex < 0 ? -1 : (stripRow.centerIndex + offset)
                        readonly property bool outOfRange: (computedFrameIndex < 0) || (computedFrameIndex >= stripRow.mediaFrameCount)
                        readonly property bool isCenter: offset === 0
                        // Use Python slots for Fraction-precision frame checks
                        // Reference cutRanges/selectionRange to trigger re-evaluation when they change
                        readonly property bool isCut: stripRow.cutRanges.length >= 0 && stripRow.appController ? stripRow.appController.isFrameCut(computedFrameIndex) : false
                        readonly property bool inSelection: stripRow.selectionRange.length >= 0 && stripRow.appController ? stripRow.appController.isFrameInSelection(computedFrameIndex) : false

                        Layout.preferredWidth: stripRow.frameWidth
                        Layout.preferredHeight: stripRow.frameHeight
                        radius: Theme.panelRadius - 2
                        color: outOfRange ? "transparent" : Theme.surface
                        border.color: outOfRange
                                      ? "transparent"
                                      : (isCenter ? Theme.accent
                                                  : (frameHoverArea.containsMouse ? Theme.textSecondary : Theme.surfaceBorder))
                        border.width: outOfRange ? 0 : (isCenter ? 2 : 1)
                        opacity: outOfRange ? 0.0 : 1.0

                        MouseArea {
                            id: frameHoverArea
                            anchors.fill: parent
                            hoverEnabled: true
                            cursorShape: Qt.PointingHandCursor
                            enabled: !frameDelegate.outOfRange
                            onClicked: {
                                if (stripRow.appController && stripRow.appController.hasVideo) {
                                    stripRow.appController.seekToTime(stripRow.appController.frameTimeForIndex(frameDelegate.computedFrameIndex))
                                    if (stripRow.appController.hintManager) {
                                        stripRow.appController.hintManager.dismissHint("frame_strip")
                                    }
                                }
                            }
                        }

                        VideoTextureItem {
                            id: thumbImage
                            anchors.fill: parent
                            anchors.margins: Theme.spacingXs
                            frameCache: stripRow.frameCache
                            playbackController: stripRow.playbackController
                            // Provide -1 when out of range to ensure the item clears
                            frameIndex: frameDelegate.outOfRange ? -1 : frameDelegate.computedFrameIndex
                            visible: !frameDelegate.outOfRange
                        }

                        Label {
                            anchors.centerIn: parent
                            text: qsTr("Loading…")
                            visible: !frameDelegate.outOfRange && (!thumbImage || !thumbImage.hasFrame)
                            color: Theme.textSecondary
                        }

                        // Selection overlay (below cut overlay)
                        Rectangle {
                            anchors.fill: parent
                            anchors.margins: Theme.spacingXs
                            radius: frameDelegate.radius - 2
                            color: Theme.selectionHighlight
                            border.color: Theme.timelineSelectionBorder
                            border.width: 1
                            visible: frameDelegate.inSelection && !frameDelegate.outOfRange && !frameDelegate.isCut
                            z: 1
                        }

                        // Cut overlay (takes precedence over selection)
                        Rectangle {
                            anchors.fill: parent
                            anchors.margins: Theme.spacingXs
                            radius: frameDelegate.radius - 2
                            color: Theme.timelineCutOverlay
                            border.color: Theme.timelineCutOverlayBorder
                            border.width: 1
                            visible: frameDelegate.isCut && !frameDelegate.outOfRange
                            z: 2
                        }
                    }
                }
            }
        }
    }
}
