import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

RowLayout {
    id: root
    spacing: Theme.spacingMd
    Layout.alignment: Qt.AlignHCenter

    required property var appWorkspaceController
    readonly property var appController: root.appWorkspaceController
    readonly property var timelineBackend: appController.timelineBackend
    // Use mutable properties with explicit signal connections (QVariantList bindings don't auto-update)
    property var selectionRange: []
    property var loopRange: []
    property bool loopActive: false

    // Update properties when signals fire (QVariantList properties need explicit updates)
    Connections {
        target: root.timelineBackend
        function onSelectionChanged() {
            root.selectionRange = root.timelineBackend ? root.timelineBackend.selectionRange : []
        }
    }
    Connections {
        target: root.appController
        function onLoopRangeChanged() {
            root.loopRange = root.appController.loopRange
        }
        function onLoopEnabledChanged() {
            root.loopActive = root.appController.loopEnabled
        }
    }
    onTimelineBackendChanged: {
        selectionRange = timelineBackend ? timelineBackend.selectionRange : []
    }

    // Initialize loop state from appController (required property is always set)
    Component.onCompleted: {
        loopRange = appController.loopRange
        loopActive = appController.loopEnabled
    }

    readonly property bool controlsEnabled: appController.mediaStatus === "ready"
    readonly property real playbackSpeed: appController.playbackSpeed
    readonly property bool audioEnabled: appController.audioEnabled
    readonly property real masterVolume: appController.masterVolume
    readonly property bool muted: appController.muted

    // Seek settings for tooltips
    readonly property var seekSettings: appController.settingsModel
                                        ? appController.settingsModel.seek_settings()
                                        : { small_amount: 1, small_unit: "frames", medium_amount: 3, medium_unit: "seconds", large_amount: 30, large_unit: "seconds" }

    // Playback speed options from settings model
    readonly property var speedOptions: appController.settingsModel
                                        ? appController.settingsModel.playback_speeds()
                                        : [0.25, 0.5, 0.75, 1.0, 1.25, 1.5, 1.75, 2.0, 2.25, 2.5, 2.75, 3.0, 3.25, 3.5, 3.75, 4.0, 8.0]

    function formatSeekTooltip(direction: string, size: string): string {
        var amount = seekSettings[size + "_amount"]
        var unit = seekSettings[size + "_unit"]
        var unitLabel = (unit === "frames")
            ? (amount === 1 ? qsTr("frame") : qsTr("frames"))
            : (amount === 1 ? qsTr("second") : qsTr("seconds"))
        return direction + " " + amount + " " + unitLabel
    }

    ComboBox {
        id: speedControl
        Layout.preferredWidth: 90
        Layout.preferredHeight: 30
        focusPolicy: Qt.TabFocus
        enabled: root.controlsEnabled

        // Build model from speedOptions
        model: {
            var result = []
            var speeds = root.speedOptions
            for (var i = 0; i < speeds.length; i++) {
                var v = speeds[i]
                // Format label: show integer for whole numbers, otherwise show decimals
                var label = (v === Math.floor(v)) ? v.toString() + "×" : v.toString() + "×"
                result.push({ value: v, label: label })
            }
            return result
        }

        textRole: "label"
        valueRole: "value"

        currentIndex: {
            for (var i = 0; i < model.length; i++) {
                if (Math.abs(model[i].value - root.playbackSpeed) < 0.01) {
                    return i
                }
            }
            // Default to 1× if not found
            for (var j = 0; j < model.length; j++) {
                if (Math.abs(model[j].value - 1.0) < 0.01) {
                    return j
                }
            }
            return 0
        }

        onActivated: function(index) {
            if (index >= 0 && index < model.length) {
                root.appController.setPlaybackSpeed(model[index].value)
            }
        }

        ToolTip.visible: hovered
        ToolTip.text: qsTr("Playback speed")
    }

    // Playback button cluster
    Item {
        Layout.preferredHeight: 52
        implicitWidth: buttonCluster.implicitWidth + 8
        implicitHeight: 52

        // Shorter pill behind buttons - play button extends beyond
        Rectangle {
            anchors.centerIn: parent
            width: parent.width
            height: 36
            color: Theme.surface
            border.width: 2
            border.color: Theme.surfaceBorder
            radius: 18
        }

        RowLayout {
            id: buttonCluster
            anchors.centerIn: parent
            spacing: 0

            // Large back
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("arrow-big-left-lines")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("large", -1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Back"), "large")
            }

            // Medium back
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("player-track-prev")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("medium", -1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Back"), "medium")
            }

            // Small back
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("arrow-badge-left")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("small", -1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Back"), "small")
            }

            // Play/Pause - larger, prominent
            RoundButton {
                id: playButton
                objectName: "playButton"
                implicitWidth: 48
                implicitHeight: 48
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                display: AbstractButton.IconOnly
                icon.source: root.appController.playing
                             ? Theme.iconUrl("player-pause")
                             : Theme.iconUrl("player-play")
                icon.width: 26
                icon.height: 26
                icon.color: enabled ? Theme.textOnAccent : Theme.textSecondary
                onClicked: root.appController.togglePlayback()
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.appController.playing ? qsTr("Pause") : qsTr("Play")

                // background: Rectangle {
                //     radius: width / 2
                //     color: playButton.pressed ? Theme.accent : (playButton.hovered ? Qt.lighter(Theme.surfaceRaised, 1.3) : "transparent")
                //     border.width: 2
                //     border.color: Theme.accent
                // }

                palette.button: Theme.accentStrong
                // border.color: Theme.accent
            }

            // Small forward
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("arrow-badge-right")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("small", 1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Forward"), "small")
            }

            // Medium forward
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("player-track-next")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("medium", 1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Forward"), "medium")
            }

            // Large forward
            RoundButton {
                implicitWidth: 36
                implicitHeight: 44
                focusPolicy: Qt.TabFocus
                enabled: root.controlsEnabled
                flat: true
                display: AbstractButton.IconOnly
                icon.source: Theme.iconUrl("arrow-big-right-lines")
                icon.width: 18
                icon.height: 18
                icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
                onClicked: root.appController.seekRelative("large", 1)
                ToolTip.visible: hovered
                ToolTip.delay: 200
                ToolTip.text: root.formatSeekTooltip(qsTr("Forward"), "large")
            }
        }
    }

    // Skip cuts toggle: play through cuts vs skip to end of cuts
    RoundButton {
        objectName: "skipCutsToggle"
        Layout.preferredWidth: 44
        Layout.preferredHeight: 44
        implicitWidth: 44
        implicitHeight: 44
        focusPolicy: Qt.TabFocus
        enabled: root.controlsEnabled
        display: AbstractButton.IconOnly
        icon.source: root.appController.skipCutsEnabled
                     ? Theme.iconUrl("scissors-off")
                     : Theme.iconUrl("scissors")
        icon.width: 22
        icon.height: 22
        icon.color: root.appController.skipCutsEnabled
                    ? Theme.accent
                    : (enabled ? Theme.textPrimary : Theme.textSecondary)
        ToolTip.visible: hovered
        ToolTip.text: root.appController.skipCutsEnabled
                      ? qsTr("Play through cuts")
                      : qsTr("Skip cuts during playback")
        onClicked: {
            root.appController.toggleSkipCuts()
            if (root.appController.skipCutsEnabled && root.appController.hintManager) {
                root.appController.hintManager.dismissHint("skip_cuts_toggle")
            }
        }
    }

    // Loop toggle: if active → clear; otherwise set loop to current selection
    RoundButton {
        objectName: "loopToggle"
        Layout.preferredWidth: 44
        Layout.preferredHeight: 44
        implicitWidth: 44
        implicitHeight: 44
        focusPolicy: Qt.TabFocus
        enabled: root.controlsEnabled && (root.loopActive || (root.selectionRange && root.selectionRange.length === 2 && Math.abs(root.selectionRange[1] - root.selectionRange[0]) > 0.001))
        display: AbstractButton.IconOnly
        icon.source: (root.loopActive ? Theme.iconUrl("repeat-off") : Theme.iconUrl("repeat"))
        icon.width: 22
        icon.height: 22
        icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
        ToolTip.visible: hovered
        ToolTip.text: root.loopActive ? qsTr("Clear loop") : qsTr("Loop selection")
        onClicked: {
            if (root.loopActive) {
                root.appController.clearLoop()
            } else if (root.selectionRange && root.selectionRange.length === 2) {
                var a = Math.min(root.selectionRange[0], root.selectionRange[1])
                var b = Math.max(root.selectionRange[0], root.selectionRange[1])
                root.appController.loopSelectionRange(a, b)
            }
        }
    }

    RowLayout {
        Layout.fillWidth: true
        Layout.alignment: Qt.AlignVCenter
        spacing: Theme.spacingSm

        RoundButton {
            Layout.preferredWidth: 44
            Layout.preferredHeight: 44
            implicitWidth: 44
            implicitHeight: 44
            enabled: root.controlsEnabled && root.audioEnabled
            display: AbstractButton.IconOnly
            focusPolicy: Qt.TabFocus
            icon.width: 22
            icon.height: 22
            icon.source: root.muted
                         ? Theme.iconUrl("volume-off")
                         : (root.masterVolume <= 0.001)
                           ? Theme.iconUrl("volume-3")
                           : (root.masterVolume < 0.33)
                             ? Theme.iconUrl("volume-2")
                             : Theme.iconUrl("volume")
            icon.color: enabled ? Theme.textPrimary : Theme.textSecondary
            onClicked: root.appController.toggleMuted()
        }

        Slider {
            Layout.fillWidth: true
            Layout.maximumWidth: 180
            Layout.minimumWidth: 100
            enabled: root.controlsEnabled && root.audioEnabled
            from: 0
            to: 100
            value: root.audioEnabled ? root.masterVolume * 100.0 : 100.0
            stepSize: 1
            onMoved: root.appController.setMasterVolume(value / 100.0)
            // Show current volume percentage only while dragging
            ToolTip.visible: pressed
            ToolTip.text: Math.round(value) + "%"
        }
    }
}
