pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import styles 1.0

Item {
    id: root

    property var analysisController: null
    property bool showOverlay: analysisController && analysisController.showDetectionOverlay

    // Current detection area as percentages (0-100)
    readonly property real topPad: analysisController ? analysisController.detectionAreaTop : 0
    readonly property real leftPad: analysisController ? analysisController.detectionAreaLeft : 0
    readonly property real rightPad: analysisController ? analysisController.detectionAreaRight : 0
    readonly property real bottomPad: analysisController ? analysisController.detectionAreaBottom : 0

    visible: root.showOverlay
    clip: false  // Allow handles to extend beyond bounds

    // Computed pixel positions for the active region
    readonly property real activeX: width * (leftPad / 100)
    readonly property real activeY: height * (topPad / 100)
    readonly property real activeWidth: width * (1 - (leftPad + rightPad) / 100)
    readonly property real activeHeight: height * (1 - (topPad + bottomPad) / 100)

    // Dimmed regions outside the detection area
    Rectangle {
        id: topDim
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: parent.top
        height: root.activeY
        color: Theme.detectionOverlayDim
    }

    Rectangle {
        id: bottomDim
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.bottom: parent.bottom
        height: root.height - root.activeY - root.activeHeight
        color: Theme.detectionOverlayDim
    }

    Rectangle {
        id: leftDim
        anchors.left: parent.left
        anchors.top: topDim.bottom
        anchors.bottom: bottomDim.top
        width: root.activeX
        color: Theme.detectionOverlayDim
    }

    Rectangle {
        id: rightDim
        anchors.right: parent.right
        anchors.top: topDim.bottom
        anchors.bottom: bottomDim.top
        width: root.width - root.activeX - root.activeWidth
        color: Theme.detectionOverlayDim
    }

    // Active detection region border
    Rectangle {
        id: activeRegion
        x: root.activeX
        y: root.activeY
        width: root.activeWidth
        height: root.activeHeight
        color: "transparent"
        border.color: Theme.detectionAreaBorder
        border.width: 2
    }

    // Handle size
    readonly property real handleSize: 12

    // Corner handles
    Repeater {
        model: [
            { corner: "topLeft", cursorShape: Qt.SizeFDiagCursor },
            { corner: "topRight", cursorShape: Qt.SizeBDiagCursor },
            { corner: "bottomLeft", cursorShape: Qt.SizeBDiagCursor },
            { corner: "bottomRight", cursorShape: Qt.SizeFDiagCursor }
        ]
        delegate: Rectangle {
            id: cornerHandle
            required property var modelData
            required property int index

            z: 20
            width: root.handleSize
            height: root.handleSize
            radius: 2
            color: Theme.detectionAreaBorder
            border.color: Theme.surface
            border.width: 1

            x: {
                if (modelData.corner === "topLeft" || modelData.corner === "bottomLeft")
                    return root.activeX - width / 2
                return root.activeX + root.activeWidth - width / 2
            }
            y: {
                if (modelData.corner === "topLeft" || modelData.corner === "topRight")
                    return root.activeY - height / 2
                return root.activeY + root.activeHeight - height / 2
            }

            MouseArea {
                anchors.fill: parent
                anchors.margins: -4  // Larger hit area
                cursorShape: cornerHandle.modelData.cursorShape

                onPositionChanged: {
                    if (!pressed || !root.analysisController || root.width <= 0 || root.height <= 0) return

                    // Map mouse position to root's coordinate space
                    let pos = mapToItem(root, mouseX, mouseY)
                    let xPct = (pos.x / root.width) * 100
                    let yPct = (pos.y / root.height) * 100

                    let newTop = root.topPad
                    let newLeft = root.leftPad
                    let newRight = root.rightPad
                    let newBottom = root.bottomPad

                    if (cornerHandle.modelData.corner === "topLeft") {
                        newTop = Math.max(0, Math.min(99 - newBottom, yPct))
                        newLeft = Math.max(0, Math.min(99 - newRight, xPct))
                    } else if (cornerHandle.modelData.corner === "topRight") {
                        newTop = Math.max(0, Math.min(99 - newBottom, yPct))
                        newRight = Math.max(0, Math.min(99 - newLeft, 100 - xPct))
                    } else if (cornerHandle.modelData.corner === "bottomLeft") {
                        newBottom = Math.max(0, Math.min(99 - newTop, 100 - yPct))
                        newLeft = Math.max(0, Math.min(99 - newRight, xPct))
                    } else {
                        newBottom = Math.max(0, Math.min(99 - newTop, 100 - yPct))
                        newRight = Math.max(0, Math.min(99 - newLeft, 100 - xPct))
                    }

                    root.analysisController.setDetectionArea(newTop, newLeft, newRight, newBottom)
                }
            }
        }
    }

    // Edge handles (centered on each edge)
    Repeater {
        model: [
            { edge: "top", cursorShape: Qt.SizeVerCursor },
            { edge: "bottom", cursorShape: Qt.SizeVerCursor },
            { edge: "left", cursorShape: Qt.SizeHorCursor },
            { edge: "right", cursorShape: Qt.SizeHorCursor }
        ]
        delegate: Rectangle {
            id: edgeHandle
            required property var modelData
            required property int index

            z: 20
            width: modelData.edge === "top" || modelData.edge === "bottom" ? root.handleSize * 2 : root.handleSize
            height: modelData.edge === "left" || modelData.edge === "right" ? root.handleSize * 2 : root.handleSize
            radius: 2
            color: Theme.detectionAreaBorder
            border.color: Theme.surface
            border.width: 1

            x: {
                if (modelData.edge === "left")
                    return root.activeX - width / 2
                if (modelData.edge === "right")
                    return root.activeX + root.activeWidth - width / 2
                return root.activeX + root.activeWidth / 2 - width / 2
            }
            y: {
                if (modelData.edge === "top")
                    return root.activeY - height / 2
                if (modelData.edge === "bottom")
                    return root.activeY + root.activeHeight - height / 2
                return root.activeY + root.activeHeight / 2 - height / 2
            }

            MouseArea {
                anchors.fill: parent
                anchors.margins: -4
                cursorShape: edgeHandle.modelData.cursorShape

                onPositionChanged: {
                    if (!pressed || !root.analysisController || root.width <= 0 || root.height <= 0) return

                    // Map mouse position to root's coordinate space
                    let pos = mapToItem(root, mouseX, mouseY)

                    let newTop = root.topPad
                    let newLeft = root.leftPad
                    let newRight = root.rightPad
                    let newBottom = root.bottomPad

                    if (edgeHandle.modelData.edge === "top") {
                        let yPct = (pos.y / root.height) * 100
                        newTop = Math.max(0, Math.min(99 - newBottom, yPct))
                    } else if (edgeHandle.modelData.edge === "bottom") {
                        let yPct = (pos.y / root.height) * 100
                        newBottom = Math.max(0, Math.min(99 - newTop, 100 - yPct))
                    } else if (edgeHandle.modelData.edge === "left") {
                        let xPct = (pos.x / root.width) * 100
                        newLeft = Math.max(0, Math.min(99 - newRight, xPct))
                    } else {
                        let xPct = (pos.x / root.width) * 100
                        newRight = Math.max(0, Math.min(99 - newLeft, 100 - xPct))
                    }

                    root.analysisController.setDetectionArea(newTop, newLeft, newRight, newBottom)
                }
            }
        }
    }

    // Percentage labels at edges (only show when padding > 0)
    Label {
        visible: root.topPad > 0
        anchors.horizontalCenter: activeRegion.horizontalCenter
        anchors.bottom: activeRegion.top
        anchors.bottomMargin: 4
        text: Math.round(root.topPad) + "%"
        font.pixelSize: 11
        color: Theme.textOnAccent
        background: Rectangle {
            color: Theme.detectionAreaBorder
            radius: 2
            anchors.fill: parent
            anchors.margins: -2
        }
    }

    Label {
        visible: root.bottomPad > 0
        anchors.horizontalCenter: activeRegion.horizontalCenter
        anchors.top: activeRegion.bottom
        anchors.topMargin: 4
        text: Math.round(root.bottomPad) + "%"
        font.pixelSize: 11
        color: Theme.textOnAccent
        background: Rectangle {
            color: Theme.detectionAreaBorder
            radius: 2
            anchors.fill: parent
            anchors.margins: -2
        }
    }

    Label {
        visible: root.leftPad > 0
        anchors.verticalCenter: activeRegion.verticalCenter
        anchors.right: activeRegion.left
        anchors.rightMargin: 4
        text: Math.round(root.leftPad) + "%"
        font.pixelSize: 11
        color: Theme.textOnAccent
        background: Rectangle {
            color: Theme.detectionAreaBorder
            radius: 2
            anchors.fill: parent
            anchors.margins: -2
        }
    }

    Label {
        visible: root.rightPad > 0
        anchors.verticalCenter: activeRegion.verticalCenter
        anchors.left: activeRegion.right
        anchors.leftMargin: 4
        text: Math.round(root.rightPad) + "%"
        font.pixelSize: 11
        color: Theme.textOnAccent
        background: Rectangle {
            color: Theme.detectionAreaBorder
            radius: 2
            anchors.fill: parent
            anchors.margins: -2
        }
    }
}
