pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import "../../../utils" as Utils

Rectangle {
    id: root

    property string label: ""
    property string filePath: ""
    property string placeholder: qsTr("Drop file here")
    property bool isCorrupt: false  // Show warning styling
    property bool interactive: true  // Can accept drops and clicks

    signal fileDropped(string path)
    signal browseClicked()

    property bool dragging: false

    color: dragging ? Theme._withAlpha(Theme.accent, 0.1) : Theme.background
    radius: Theme.panelRadius
    border.width: 2
    border.color: dragging ? Theme.accent : Theme.surfaceBorder

    Behavior on border.color {
        ColorAnimation { duration: 150 }
    }

    Behavior on color {
        ColorAnimation { duration: 150 }
    }

    DropArea {
        anchors.fill: parent
        enabled: root.interactive

        onEntered: function(drag) {
            root.dragging = true
        }
        onExited: function() {
            root.dragging = false
        }
        onDropped: function(drop) {
            root.dragging = false
            if (drop.hasUrls && drop.urls.length > 0) {
                var path = Utils.PathUtils.urlToPath(drop.urls[0])
                root.fileDropped(path)
            }
        }
    }

    MouseArea {
        anchors.fill: parent
        enabled: root.interactive && !root.filePath
        cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor
        onClicked: root.browseClicked()
    }

    ColumnLayout {
        anchors.centerIn: parent
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.leftMargin: Theme.spacingMd
        anchors.rightMargin: Theme.spacingMd
        spacing: Theme.spacingSm

        // Label
        Label {
            Layout.alignment: Qt.AlignHCenter
            text: root.label
            font.weight: Font.DemiBold
            font.pixelSize: 14
            color: Theme.textPrimary
        }

        // Icon area
        Rectangle {
            Layout.alignment: Qt.AlignHCenter
            Layout.preferredWidth: 64
            Layout.preferredHeight: 64
            radius: Theme.panelRadius
            color: root.isCorrupt
                ? Theme._withAlpha(Theme.statusDanger, 0.1)
                : (root.filePath ? Theme._withAlpha(Theme.accent, 0.1) : Theme.surfaceRaised)

            Image {
                anchors.centerIn: parent
                source: root.isCorrupt ? Theme.iconUrl("alert-triangle") : Theme.iconUrl("export")
                sourceSize.width: 32
                sourceSize.height: 32
                opacity: root.filePath ? 1.0 : 0.5
            }
        }

        // File name or placeholder
        Label {
            Layout.alignment: Qt.AlignHCenter
            Layout.fillWidth: true
            text: root.filePath ? root.filePath.split("/").pop() : root.placeholder
            elide: Text.ElideMiddle
            horizontalAlignment: Text.AlignHCenter
            wrapMode: Text.WordWrap
            maximumLineCount: 2
            color: root.filePath ? Theme.textPrimary : Theme.textSecondary
            font.pixelSize: 12
        }

        // Browse button
        Button {
            id: browseButton
            Layout.alignment: Qt.AlignHCenter
            visible: root.interactive && !root.filePath
            text: qsTr("Browse...")
            font.pixelSize: 12
            implicitHeight: 28
            padding: Theme.spacingSm
            onClicked: root.browseClicked()

            background: Rectangle {
                radius: Theme.panelRadius - 2
                color: browseButton.down ? Theme.surfaceStrong : (browseButton.hovered ? Theme.surfaceRaised : Theme.surface)
                border.width: 1
                border.color: Theme.surfaceBorder
            }
        }

        // Status indicator for set files
        Label {
            Layout.alignment: Qt.AlignHCenter
            visible: root.filePath && !root.interactive
            text: root.isCorrupt ? qsTr("Corrupt file") : qsTr("Ready")
            font.pixelSize: 11
            color: root.isCorrupt ? Theme.statusDanger : Theme.textSecondary
        }
    }
}
