pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import components.controls 1.0 as Controls
import components.forms 1.0 as Forms
import components.layout 1.0 as Layouts
import SMC.Controllers 1.0

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    signal openFeedback()

    readonly property var sections: [
        { key: "general", title: qsTr("General"), subtitle: qsTr("Appearance & playback") },
        { key: "tutorial", title: qsTr("Tutorial"), subtitle: qsTr("Guided hints") },
        { key: "shortcuts", title: qsTr("Shortcuts"), subtitle: qsTr("Keyboard mapping") },
        { key: "models", title: qsTr("Models"), subtitle: qsTr("Download sources") },
        { key: "license", title: qsTr("Cloud License"), subtitle: qsTr("Cloud transcription") },
        { key: "export", title: qsTr("Export defaults"), subtitle: qsTr("New project presets") },
        { key: "performance", title: qsTr("Performance"), subtitle: qsTr("Hardware acceleration") },
        { key: "updates", title: qsTr("Updates"), subtitle: qsTr("Update checks & privacy") }
    ]
    property int currentSectionIndex: 0
    readonly property string currentSectionKey: root.sections[root.currentSectionIndex].key

    required property var settingsModel
    required property var themeController
    required property var appWorkspaceController
    required property var languageController
    property var updateController: null
    readonly property var transcriptionController: root.appWorkspaceController.transcriptionController

    readonly property var shortcutGroups: [
        {
            title: qsTr("File"),
            shortcuts: [
                { id: "save_project", label: qsTr("Save project"), helper: qsTr("Save the current project to disk") },
                { id: "save_project_as", label: qsTr("Save project as…"), helper: qsTr("Save the current project with a new name") }
            ]
        },
        {
            title: qsTr("Playback"),
            shortcuts: [
                { id: "toggle_playback", label: qsTr("Toggle playback"), helper: qsTr("Play or pause the current media") },
                { id: "speed_up", label: qsTr("Speed up"), helper: qsTr("Increase playback speed to next preset") },
                { id: "speed_down", label: qsTr("Speed down"), helper: qsTr("Decrease playback speed to previous preset") },
                { id: "loop_selection", label: qsTr("Loop selection"), helper: qsTr("Set A/B loop to current selection") },
                { id: "clear_loop", label: qsTr("Clear loop"), helper: qsTr("Remove A/B loop markers") }
            ]
        },
        {
            title: qsTr("Seeking"),
            shortcuts: [
                { id: "seek_small_back", label: qsTr("Seek small back"), helper: qsTr("Jump backward by small step (configurable in General)") },
                { id: "seek_small_forward", label: qsTr("Seek small forward"), helper: qsTr("Jump forward by small step (configurable in General)") },
                { id: "seek_medium_back", label: qsTr("Seek medium back"), helper: qsTr("Jump backward by medium step") },
                { id: "seek_medium_forward", label: qsTr("Seek medium forward"), helper: qsTr("Jump forward by medium step") },
                { id: "seek_large_back", label: qsTr("Seek large back"), helper: qsTr("Jump backward by large step") },
                { id: "seek_large_forward", label: qsTr("Seek large forward"), helper: qsTr("Jump forward by large step") },
                { id: "zoom_in", label: qsTr("Zoom in"), helper: qsTr("Zoom in on the timeline") },
                { id: "zoom_out", label: qsTr("Zoom out"), helper: qsTr("Zoom out on the timeline") }
            ]
        },
        {
            title: qsTr("Editing"),
            shortcuts: [
                { id: "undo", label: qsTr("Undo"), helper: qsTr("Undo the last action") },
                { id: "redo", label: qsTr("Redo"), helper: qsTr("Redo the previously undone action") },
                { id: "mark_in", label: qsTr("Mark In / Start Selection"), helper: qsTr("Set selection start at current frame") },
                { id: "mark_out", label: qsTr("Mark Out / End Selection"), helper: qsTr("Set selection end at current frame") },
                { id: "cut_selection", label: qsTr("Cut Selection"), helper: qsTr("Mark the current selection as cut (removed from output)") },
                { id: "cut_unselected", label: qsTr("Cut Unselected"), helper: qsTr("Cut everything outside the selection (keep only selection)") },
                { id: "uncut_selection", label: qsTr("Uncut Selection"), helper: qsTr("Remove cuts from the current selection (restore to output)") },
                { id: "center_selection", label: qsTr("Center Selection"), helper: qsTr("Zoom and pan timeline to fit current selection") }
            ]
        }
    ]

    readonly property var videoEncoderOptions: [
        { label: qsTr("Same as source (copy)"), value: "copy" },
        { label: qsTr("H.264"), value: "h264" },
        { label: qsTr("HEVC / H.265"), value: "hevc" },
        { label: qsTr("VP9"), value: "vp9" },
        { label: qsTr("AV1"), value: "av1" }
    ]

    readonly property var videoQualityOptions: [
        { label: qsTr("Low"), value: 0 },
        { label: qsTr("Normal"), value: 1 },
        { label: qsTr("High"), value: 2 },
        { label: qsTr("Almost indistinguishable"), value: 3 },
        { label: qsTr("Near lossless"), value: 4 },
        { label: qsTr("Lossless"), value: 5 }
    ]

    readonly property var xmlModeOptions: [
        { label: "DaVinci (.otio)", value: "OTIO" },
        { label: "Final Cut Pro 7 XML", value: "FC7_XML" },
        { label: "Final Cut Pro X (.fcpxml)", value: "FCPX" },
        { label: "Avid AAF", value: "AAF" },
        { label: "Shotcut MLT", value: "MLT" }
    ]

    function pageForKey(key) {
        switch (key) {
        case "general":
        default:
            return generalPage
        case "tutorial":
            return tutorialPage
        case "shortcuts":
            return shortcutsPage
        case "models":
            return modelsPage
        case "license":
            return licensePage
        case "export":
            return exportPage
        case "performance":
            return performancePage
        case "updates":
            return updatesPage
        }
    }

    width: parent ? Math.min(parent.width * 0.9, 1040) : 960
    height: parent ? Math.min(parent.height * 0.85, 760) : 680
    anchors.centerIn: parent

    background: Rectangle {
        radius: 18
        color: "transparent"
    }

    contentItem: Rectangle {
        id: chrome
        anchors.fill: parent
        radius: 18
        color: Theme.surfaceStrong
        border.color: Theme.surfaceBorder
        border.width: 1

        ColumnLayout {
            anchors.fill: parent
            anchors.margins: Theme.spacingXl
            spacing: Theme.spacingLg

            RowLayout {
                Layout.fillWidth: true

                Label {
                    text: qsTr("Settings")
                    font.pixelSize: 30
                    font.bold: true
                    Layout.fillWidth: true
                }

                ToolButton {
                    id: closeButton
                    Accessible.name: qsTr("Close settings")
                    text: "\u2715"
                    font.pixelSize: 18
                    implicitWidth: 36
                    implicitHeight: 36
                    onClicked: root.close()
                }
            }

            RowLayout {
                Layout.fillWidth: true
                Layout.fillHeight: true
                spacing: Theme.spacingXl

                Rectangle {
                    Layout.preferredWidth: 260
                    Layout.fillHeight: true
                    color: Theme.surface

                    Rectangle {
                        anchors.right: parent.right
                        anchors.top: parent.top
                        anchors.bottom: parent.bottom
                        width: 1
                        color: Theme.surfaceBorder
                    }

                    ListView {
                        id: navList
                        anchors.fill: parent
                        anchors.margins: Theme.spacingMd
                        spacing: 4
                        clip: true
                        model: root.sections
                        currentIndex: root.currentSectionIndex
                        interactive: true

                        delegate: ItemDelegate {
                            id: navDelegate
                            required property int index
                            required property var modelData

                            width: ListView.view.width
                            height: 60
                            highlighted: ListView.view.currentIndex === navDelegate.index
                            onClicked: ListView.view.currentIndex = navDelegate.index

                            background: Rectangle {
                                radius: Theme.panelRadius
                                color: navDelegate.highlighted ? Theme.surfaceRaised : (navDelegate.hovered ? Qt.rgba(Theme.surfaceRaised.r, Theme.surfaceRaised.g, Theme.surfaceRaised.b, 0.5) : "transparent")

                                Rectangle {
                                    visible: navDelegate.highlighted
                                    anchors.left: parent.left
                                    anchors.verticalCenter: parent.verticalCenter
                                    anchors.leftMargin: 4
                                    height: 24
                                    width: 3
                                    radius: 1.5
                                    color: Theme.accent
                                }
                            }

                            contentItem: ColumnLayout {
                                anchors.fill: parent
                                anchors.leftMargin: Theme.spacingMd
                                anchors.rightMargin: Theme.spacingMd
                                spacing: 2

                                Item { Layout.fillHeight: true } // Top spacer

                                Label {
                                    text: navDelegate.modelData.title
                                    font.weight: navDelegate.highlighted ? Font.DemiBold : Font.Normal
                                    font.pixelSize: 14
                                    color: Theme.textPrimary
                                    elide: Label.ElideRight
                                    Layout.fillWidth: true
                                }

                                Label {
                                    visible: navDelegate.modelData.subtitle.length > 0
                                    text: navDelegate.modelData.subtitle
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                    elide: Label.ElideRight
                                    Layout.fillWidth: true
                                }

                                Item { Layout.fillHeight: true } // Bottom spacer
                            }
                        }
                    }

                    Connections {
                        target: navList
                        function onCurrentIndexChanged() {
                            if (root.currentSectionIndex !== navList.currentIndex) {
                                root.currentSectionIndex = navList.currentIndex
                            }
                        }
                    }
                }

                Loader {
                    id: pageLoader
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    sourceComponent: root.pageForKey(root.currentSectionKey)
                }
            }

            RowLayout {
                Layout.fillWidth: true
                Layout.alignment: Qt.AlignRight

                Controls.SecondaryButton {
                    text: qsTr("Close")
                    onClicked: root.close()
                }
            }
        }
    }

    Component {
        id: generalPage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                id: settingsColumn
                width: parent.width
                spacing: Theme.spacingLg

                Layouts.SectionHeader {
                    title: qsTr("Appearance")
                    subtitle: qsTr("Configure theme and localization")
                }

                Forms.LabeledField {
                    label: qsTr("Theme mode")
                    helperText: qsTr("Choose between light and dark. First launch follows your system theme.")

                    ComboBox {
                        id: themeModeCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: [
                            { label: qsTr("Light"), value: "light" },
                            { label: qsTr("Dark"), value: "dark" }
                        ]

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        property ThemeController themeControllerRef: root.themeController ? root.themeController : null

                        Component.onCompleted: {
                            const modeValue = themeControllerRef ? themeControllerRef.mode : (root.settingsModel ? root.settingsModel.theme : "light")
                            setCurrentValue(modeValue)
                        }

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) {
                                return
                            }
                            themeControllerRef ? themeControllerRef.setMode(option.value) : root.settingsModel.set_theme(option.value)
                        }

                        Connections {
                            target: root.themeController || null
                            enabled: !!root.themeController
                            ignoreUnknownSignals: true
                            function onModeChanged(mode) {
                                themeModeCombo.setCurrentValue(mode)
                            }
                        }
                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onTheme_changed(mode) {
                                themeModeCombo.setCurrentValue(mode)
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Language")
                    helperText: qsTr("Application interface language. Changes take effect immediately.")

                    ComboBox {
                        id: languageCombo
                        Layout.fillWidth: true
                        textRole: "name"
                        valueRole: "code"
                        model: root.languageController ? root.languageController.availableLanguages : []

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].code === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: {
                            if (root.languageController) {
                                setCurrentValue(root.languageController.currentLanguage)
                            }
                        }

                        onActivated: function(index) {
                            const option = model[index]
                            if (option && root.languageController) {
                                root.languageController.setLanguage(option.code)
                            }
                        }

                        Connections {
                            target: root.languageController || null
                            ignoreUnknownSignals: true
                            function onLanguageChanged(lang) {
                                languageCombo.setCurrentValue(lang)
                            }
                        }
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Transcript")
                    subtitle: qsTr("Transcription display and defaults")
                }

                Forms.LabeledField {
                    label: qsTr("Font size")
                    helperText: qsTr("Use Ctrl+/- to adjust while viewing transcript.")

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        TextField {
                            id: fontSizeField
                            Layout.preferredWidth: 80
                            text: root.settingsModel ? root.settingsModel.transcript_font_size() : "14"
                            validator: IntValidator { bottom: 8; top: 72 }
                            selectByMouse: true
                            onEditingFinished: {
                                if (!root.settingsModel) return
                                var val = parseInt(text)
                                if (!isNaN(val)) {
                                    root.settingsModel.set_transcript_font_size(val)
                                }
                            }

                            Connections {
                                target: root.settingsModel || null
                                ignoreUnknownSignals: true
                                function onTranscript_font_size_changed(size) {
                                    fontSizeField.text = size.toString()
                                }
                            }
                        }

                        Label {
                            text: qsTr("px")
                            color: Theme.textSecondary
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Default transcription language")
                    helperText: qsTr("Language for multilingual Whisper models. English-only models ignore this setting.")

                    Forms.LanguagePickerField {
                        id: defaultTranscriptionLanguagePicker
                        Layout.fillWidth: true
                        settingsModel: root.settingsModel
                        includeAutonyms: true
                        accessibleName: qsTr("Default transcription language")

                        function syncFromSettings() {
                            if (root.settingsModel) {
                                var savedLang = root.settingsModel.default_transcription_language()
                                defaultTranscriptionLanguagePicker.code = savedLang
                            }
                        }

                        Component.onCompleted: syncFromSettings()

                        onCodeSelected: function(langCode) {
                            if (root.settingsModel) {
                                root.settingsModel.set_default_transcription_language(langCode)
                            }
                        }

                        Connections {
                            target: root
                            function onSettingsModelChanged() {
                                defaultTranscriptionLanguagePicker.syncFromSettings()
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onDefault_transcription_language_changed(langCode) {
                                defaultTranscriptionLanguagePicker.code = langCode
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Default track selection")
                    helperText: qsTr("Initial selection when opening transcription or analysis tools.")

                    ComboBox {
                        id: defaultTrackModeCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: [
                            { label: qsTr("All audio tracks"), value: "all" },
                            { label: qsTr("Only first audio track"), value: "first" }
                        ]

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: {
                            if (root.settingsModel) {
                                setCurrentValue(root.settingsModel.default_track_mode())
                            }
                        }

                        onActivated: function(index) {
                            const option = model[index]
                            if (option && root.settingsModel) {
                                root.settingsModel.set_default_track_mode(option.value)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onDefault_track_mode_changed(mode) {
                                defaultTrackModeCombo.setCurrentValue(mode)
                            }
                        }
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Playback")
                    subtitle: qsTr("Seek behavior")
                }

                QtObject {
                    id: seekSettingsState
                    property var values: root.settingsModel ? root.settingsModel.seek_settings() : {}
                }

                Connections {
                    target: root.settingsModel || null
                    ignoreUnknownSignals: true
                    function onSeek_settings_changed() {
                        if (root.settingsModel) {
                            seekSettingsState.values = root.settingsModel.seek_settings()
                        }
                    }
                }

                Repeater {
                    model: [
                        { label: qsTr("Small step"), key: "small" },
                        { label: qsTr("Medium step"), key: "medium" },
                        { label: qsTr("Large step"), key: "large" }
                    ]
                    delegate: Forms.LabeledField {
                        id: seekFieldDelegate
                        required property var modelData
                        label: modelData.label
                        helperText: qsTr("Amount and unit for the %1 seek buttons").arg(modelData.label.toLowerCase())

                        RowLayout {
                            Layout.fillWidth: true
                            spacing: Theme.spacingSm

                            TextField {
                                id: amountField
                                Layout.preferredWidth: 80
                                text: seekSettingsState.values[seekFieldDelegate.modelData.key + "_amount"] ?? ""
                                validator: DoubleValidator { bottom: 0.001; decimals: 3 }
                                selectByMouse: true
                                onEditingFinished: {
                                    if (!root.settingsModel) return
                                    var val = parseFloat(text)
                                    if (!isNaN(val)) {
                                        var current = seekSettingsState.values
                                        var args = [
                                            current.small_amount, current.small_unit,
                                            current.medium_amount, current.medium_unit,
                                            current.large_amount, current.large_unit
                                        ]
                                        // Update specific value
                                        if (seekFieldDelegate.modelData.key === "small") args[0] = val
                                        else if (seekFieldDelegate.modelData.key === "medium") args[2] = val
                                        else if (seekFieldDelegate.modelData.key === "large") args[4] = val

                                        root.settingsModel.set_seek_settings.apply(root.settingsModel, args)
                                    }
                                }
                            }

                            ComboBox {
                                id: unitCombo
                                Layout.fillWidth: true
                                textRole: "label"
                                valueRole: "value"
                                model: [
                                    { label: qsTr("Frames"), value: "frames" },
                                    { label: qsTr("Seconds"), value: "seconds" }
                                ]

                                function setCurrentValue(val: string): void {
                                    for (var i = 0; i < model.length; i++) {
                                        if (model[i].value === val) {
                                            currentIndex = i
                                            return
                                        }
                                    }
                                }

                                Component.onCompleted: {
                                    var val = seekSettingsState.values[seekFieldDelegate.modelData.key + "_unit"]
                                    if (val !== undefined) setCurrentValue(val)
                                }

                                Connections {
                                    target: seekSettingsState
                                    function onValuesChanged() {
                                        var val = seekSettingsState.values[seekFieldDelegate.modelData.key + "_unit"]
                                        if (val !== undefined) unitCombo.setCurrentValue(val)
                                    }
                                }

                                onActivated: function(index) {
                                    if (!root.settingsModel) return
                                    var val = model[index].value
                                    var current = seekSettingsState.values
                                    var args = [
                                        current.small_amount, current.small_unit,
                                        current.medium_amount, current.medium_unit,
                                        current.large_amount, current.large_unit
                                    ]
                                    // Update specific value
                                    if (seekFieldDelegate.modelData.key === "small") args[1] = val
                                    else if (seekFieldDelegate.modelData.key === "medium") args[3] = val
                                    else if (seekFieldDelegate.modelData.key === "large") args[5] = val

                                    root.settingsModel.set_seek_settings.apply(root.settingsModel, args)
                                }
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: tutorialPage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                id: tutorialColumn
                width: parent.width
                spacing: Theme.spacingLg

                readonly property var hintManager: root.appWorkspaceController ? root.appWorkspaceController.hintManager : null

                Layouts.SectionHeader {
                    title: qsTr("Tutorial Progress")
                    subtitle: qsTr("Learn the app with guided hints")
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Show tutorial hints")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Display helpful tips as you learn the app")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Switch {
                        id: tutorialHintsSwitch
                        checked: root.settingsModel ? root.settingsModel.hints_enabled() : true
                        onToggled: {
                            if (root.settingsModel) {
                                root.settingsModel.set_hints_enabled(checked)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onHints_enabled_changed(enabled) {
                                if (tutorialHintsSwitch.checked !== enabled) {
                                    tutorialHintsSwitch.checked = enabled
                                }
                            }
                        }
                    }
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    Controls.SecondaryButton {
                        text: qsTr("Reset tutorial progress")
                        enabled: root.settingsModel !== null
                        onClicked: {
                            if (root.settingsModel) {
                                root.settingsModel.reset_milestones()
                            }
                        }
                    }

                    Label {
                        id: progressLabel
                        readonly property int completed: tutorialColumn.hintManager ? tutorialColumn.hintManager.completedMilestones : 0
                        readonly property int total: tutorialColumn.hintManager ? tutorialColumn.hintManager.totalMilestones : 0
                        property bool showingMissing: false
                        text: showingMissing
                            ? (tutorialColumn.hintManager ? tutorialColumn.hintManager.getMissingMilestones().join(", ") : "")
                            : qsTr("Progress: %1/%2").arg(completed).arg(total)
                        font.pixelSize: 12
                        color: completed === total && total > 0 ? Theme.statusSuccess : Theme.textSecondary
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true

                        MouseArea {
                            anchors.fill: parent
                            onDoubleClicked: progressLabel.showingMissing = !progressLabel.showingMissing
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: shortcutsPage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                // Keyset selector
                Layouts.SectionHeader {
                    title: qsTr("Keyset")
                    subtitle: qsTr("Apply preset shortcuts from popular editors")
                }

                QtObject {
                    id: keysetState
                    property var keysets: root.settingsModel ? root.settingsModel.available_keysets() : []
                    property int selectedIndex: 0
                    property string activeKeyset: root.settingsModel ? root.settingsModel.active_keyset() : ""
                    property bool isModified: root.settingsModel ? root.settingsModel.is_keyset_modified() : false

                    function updateSelection() {
                        // Select the active keyset in the combo
                        for (var i = 0; i < keysets.length; i++) {
                            if (keysets[i].name === activeKeyset) {
                                selectedIndex = i
                                return
                            }
                        }
                    }
                }

                Connections {
                    target: root.settingsModel || null
                    function onKeysets_changed() {
                        keysetState.keysets = root.settingsModel.available_keysets()
                        keysetState.updateSelection()
                    }
                    function onActive_keyset_changed(name, modified) {
                        keysetState.activeKeyset = name
                        keysetState.isModified = modified
                        keysetState.updateSelection()
                    }
                }

                Label {
                    text: keysetState.isModified
                        ? qsTr("Current: %1 (modified)").arg(keysetState.activeKeyset)
                        : qsTr("Current: %1").arg(keysetState.activeKeyset)
                    color: keysetState.isModified ? Theme.statusDanger : Theme.textSecondary
                    font.pixelSize: 12
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ComboBox {
                        id: keysetCombo
                        Layout.fillWidth: true
                        Layout.maximumWidth: 300
                        textRole: "name"
                        model: keysetState.keysets

                        currentIndex: keysetState.selectedIndex
                        onCurrentIndexChanged: keysetState.selectedIndex = currentIndex

                        delegate: ItemDelegate {
                            id: keysetDelegate
                            required property var modelData
                            required property int index
                            width: keysetCombo.width
                            text: modelData.name
                            font.weight: modelData.builtin ? Font.Normal : Font.Medium
                            highlighted: keysetCombo.highlightedIndex === index

                            contentItem: RowLayout {
                                spacing: Theme.spacingSm
                                Label {
                                    text: keysetDelegate.modelData.name
                                    Layout.fillWidth: true
                                    elide: Text.ElideRight
                                }
                                Label {
                                    visible: !keysetDelegate.modelData.builtin
                                    text: qsTr("(custom)")
                                    color: Theme.textSecondary
                                    font.pixelSize: 11
                                }
                            }
                        }
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Apply")
                        enabled: keysetCombo.currentIndex >= 0 && keysetCombo.currentIndex < keysetState.keysets.length
                        onClicked: {
                            var keyset = keysetState.keysets[keysetCombo.currentIndex]
                            if (keyset && root.settingsModel) {
                                root.settingsModel.apply_keyset(keyset.name)
                            }
                        }
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Save as…")
                        onClicked: saveKeysetDialog.open()
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Delete")
                        visible: {
                            if (keysetCombo.currentIndex < 0 || keysetCombo.currentIndex >= keysetState.keysets.length) return false
                            var keyset = keysetState.keysets[keysetCombo.currentIndex]
                            return keyset && !keyset.builtin
                        }
                        onClicked: {
                            var keyset = keysetState.keysets[keysetCombo.currentIndex]
                            if (keyset && root.settingsModel) {
                                root.settingsModel.delete_keyset(keyset.name)
                            }
                        }
                    }
                }

                Dialog {
                    id: saveKeysetDialog
                    parent: root.contentItem
                    anchors.centerIn: parent  // qmllint disable Quick.layout-positioning
                    modal: true
                    title: qsTr("Save Keyset")
                    standardButtons: Dialog.Save | Dialog.Cancel

                    onAccepted: {
                        var name = keysetNameField.text.trim()
                        if (name.length > 0 && root.settingsModel) {
                            if (root.settingsModel.save_current_as_keyset(name)) {
                                keysetState.keysets = root.settingsModel.available_keysets()
                                // Select the newly created keyset
                                for (var i = 0; i < keysetState.keysets.length; i++) {
                                    if (keysetState.keysets[i].name === name) {
                                        keysetState.selectedIndex = i
                                        break
                                    }
                                }
                            }
                        }
                        keysetNameField.text = ""
                    }

                    onRejected: {
                        keysetNameField.text = ""
                    }

                    ColumnLayout {
                        spacing: Theme.spacingMd

                        Label {
                            text: qsTr("Enter a name for the new keyset:")
                        }

                        TextField {
                            id: keysetNameField
                            Layout.fillWidth: true
                            Layout.preferredWidth: 300
                            placeholderText: qsTr("My Custom Keyset")
                            selectByMouse: true
                        }

                        Label {
                            visible: keysetNameField.text.trim().length > 0 && root.settingsModel && root.settingsModel.is_builtin_keyset(keysetNameField.text.trim())
                            text: qsTr("Cannot use a built-in keyset name")
                            color: Theme.statusDanger
                            font.pixelSize: 11
                        }
                    }
                }

                MenuSeparator {
                    Layout.fillWidth: true
                }

                Repeater {
                    model: root.shortcutGroups
                    delegate: ColumnLayout {
                        id: groupDelegate
                        required property var modelData

                        Layout.fillWidth: true
                        spacing: Theme.spacingMd

                        Layouts.SectionHeader {
                            title: groupDelegate.modelData.title
                            subtitle: groupDelegate.modelData.subtitle || ""
                        }

                        Repeater {
                            model: groupDelegate.modelData.shortcuts
                            delegate: ColumnLayout {
                                id: shortcutDelegate
                                required property var modelData

                                Layout.fillWidth: true
                                spacing: Theme.spacingXs

                                ShortcutEditor {
                                    Layout.fillWidth: true
                                    settingsModel: root.settingsModel
                                    label: shortcutDelegate.modelData.label
                                    shortcutId: shortcutDelegate.modelData.id
                                }

                                Label {
                                    visible: shortcutDelegate.modelData.helper && shortcutDelegate.modelData.helper.length > 0
                                    text: shortcutDelegate.modelData.helper
                                    font.pixelSize: 11
                                    color: Theme.textSecondary
                                    wrapMode: Text.WordWrap
                                    Layout.leftMargin: 180
                                }
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: modelsPage

        ScrollView {
            id: modelsScrollView
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            readonly property var modelSourceOptions: [
                { key: "huggingface", label: "HuggingFace (default)" },
                { key: "smc_staging", label: "smartmediacutter.com (staging)" },
                { key: "smc_production", label: "smartmediacutter.com" }
            ]

            readonly property var modelSourceOptionsWithNone: [
                { key: "", label: qsTr("None") },
                { key: "huggingface", label: "HuggingFace" },
                { key: "smc_staging", label: "smartmediacutter.com (staging)" },
                { key: "smc_production", label: "smartmediacutter.com" }
            ]

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                Layouts.SectionHeader {
                    title: qsTr("Transcription models")
                    subtitle: qsTr("Download sources and storage")
                }

                Forms.LabeledField {
                    label: qsTr("Primary download source")
                    helperText: qsTr("Where to download Whisper models from.")

                    ComboBox {
                        id: primarySourceCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "key"
                        model: modelsScrollView.modelSourceOptions

                        function setCurrentValue(value: string): void {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].key === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: setCurrentValue(root.settingsModel.model_source_primary())

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) return
                            root.settingsModel.set_model_source_primary(option.key)
                        }

                        Connections {
                            target: root.settingsModel
                            function onModel_source_changed() {
                                primarySourceCombo.setCurrentValue(root.settingsModel.model_source_primary())
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Fallback download source")
                    helperText: qsTr("Used if the primary source fails.")

                    ComboBox {
                        id: fallbackSourceCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "key"
                        model: modelsScrollView.modelSourceOptionsWithNone

                        function setCurrentValue(value: string): void {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].key === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: setCurrentValue(root.settingsModel.model_source_fallback())

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) return
                            root.settingsModel.set_model_source_fallback(option.key)
                        }

                        Connections {
                            target: root.settingsModel
                            function onModel_source_changed() {
                                fallbackSourceCombo.setCurrentValue(root.settingsModel.model_source_fallback())
                            }
                        }
                    }
                }

                ColumnLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingSm

                    Label {
                        text: root.transcriptionController ? root.transcriptionController.downloadMessage : ""
                        visible: text.length > 0
                        wrapMode: Text.WordWrap
                        color: Theme.textSecondary
                    }

                    ProgressBar {
                        Layout.fillWidth: true
                        visible: root.transcriptionController && root.transcriptionController.progress > 0 && root.transcriptionController.progress < 100
                        value: root.transcriptionController ? root.transcriptionController.progress / 100.0 : 0
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Models folder")
                    helperText: qsTr("Downloaded Whisper checkpoints are stored here. You can delete any of them manually if storage runs low. They will be re-downloaded automatically as needed.")

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        TextArea {
                            Layout.fillWidth: true
                            readOnly: true
                            wrapMode: TextArea.WrapAnywhere
                            background: null
                            text: root.transcriptionController ? root.transcriptionController.modelsDirectory : qsTr("Not initialised yet")
                        }

                        Controls.SecondaryButton {
                            text: qsTr("Reveal in file browser")
                            enabled: !!root.transcriptionController
                            onClicked: root.transcriptionController ? root.transcriptionController.reveal_models_directory() : null
                        }
                    }
                }

                // Legacy models cleanup section
                Forms.LabeledField {
                    visible: root.transcriptionController && root.transcriptionController.hasLegacyModelsFolder
                    label: qsTr("Legacy models")
                    helperText: qsTr("An old whisper_models folder was found. These models are not compatible with the current version and can be safely deleted to free up disk space.")

                    Button {
                        Layout.fillWidth: true
                        text: qsTr("Delete Legacy Models")
                        palette.button: "#c62828"
                        palette.buttonText: "#ffffff"
                        onClicked: {
                            if (root.transcriptionController) {
                                root.transcriptionController.delete_legacy_models_folder()
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: licensePage

        ScrollView {
            id: licenseScrollView
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            // Maps error codes from LicenseManager to localized user-facing strings
            function licenseErrorMessage(errorCode: string): string {
                switch (errorCode) {
                    case "key_required":
                        return qsTr("Please enter a license key")
                    case "key_not_found":
                        return qsTr("License key not found. Please check your key and try again.")
                    case "key_expired":
                        return qsTr("This license key has expired.")
                    case "key_usage_limit":
                        return qsTr("This license key has reached its activation limit. Please contact support at info@smartmediacutter.com.")
                    case "server_error":
                        return qsTr("Server error. Please try again later.")
                    case "network_error":
                        return qsTr("Network error. Please check your internet connection.")
                    case "timeout":
                        return qsTr("Connection timed out. Please try again.")
                    case "invalid_response":
                        return qsTr("Invalid server response. Please try again later.")
                    case "token_save_failed":
                        return qsTr("Failed to save license. Please try again.")
                    default:
                        return qsTr("Activation failed. Please try again.")
                }
            }

            readonly property var licenseManager: root.appWorkspaceController ? root.appWorkspaceController.licenseManager : null

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                Layouts.SectionHeader {
                    title: qsTr("Premium License")
                    subtitle: qsTr("Activate premium features and cloud transcription services")
                }

                QtObject {
                    id: licenseState
                    property bool isValid: licenseScrollView.licenseManager ? licenseScrollView.licenseManager.isValid() : false
                    property bool isActivating: false
                    property string errorMessage: ""
                    property string successMessage: ""
                }

                Connections {
                    target: licenseScrollView.licenseManager
                    function onLicense_status_changed(valid) {
                        licenseState.isValid = valid
                    }
                    function onActivation_finished(errorCode) {
                        licenseState.isActivating = false
                        if (errorCode === "") {
                            licenseState.successMessage = qsTr("License activated successfully!")
                            licenseKeyField.text = ""
                            licenseState.isValid = true
                        } else {
                            licenseState.errorMessage = licenseScrollView.licenseErrorMessage(errorCode)
                        }
                    }
                }

                // Status indicator
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: statusLayout.implicitHeight + Theme.spacingMd * 2
                    radius: Theme.panelRadius
                    color: licenseState.isValid ? Qt.rgba(0.2, 0.6, 0.2, 0.2) : Theme.surfaceRaised

                    RowLayout {
                        id: statusLayout
                        anchors.fill: parent
                        anchors.margins: Theme.spacingMd
                        spacing: Theme.spacingMd

                        Rectangle {
                            implicitWidth: 12
                            implicitHeight: 12
                            radius: 6
                            color: licenseState.isValid ? "#4CAF50" : Theme.textSecondary
                        }

                        ColumnLayout {
                            Layout.fillWidth: true
                            spacing: 2

                            Label {
                                text: licenseState.isValid ? qsTr("Cloud license active") : qsTr("No cloud license")
                                font.pixelSize: 14
                                font.weight: Font.DemiBold
                                color: Theme.textPrimary
                            }

                            Label {
                                text: licenseState.isValid
                                    ? qsTr("Cloud transcription is enabled for this device")
                                    : qsTr("Enter your license key to activate cloud features")
                                font.pixelSize: 12
                                color: Theme.textSecondary
                            }
                        }
                    }
                }

                // Credits section (only when license is valid)
                ColumnLayout {
                    id: creditsSection
                    visible: licenseState.isValid
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    // Auto-refresh on initial load and when visibility changes
                    Component.onCompleted: {
                        if (visible && licenseScrollView.licenseManager) {
                            licenseScrollView.licenseManager.refreshCredits()
                        }
                    }
                    onVisibleChanged: {
                        if (visible && licenseScrollView.licenseManager) {
                            licenseScrollView.licenseManager.refreshCredits()
                        }
                    }

                    Layouts.SectionHeader {
                        title: qsTr("Cloud Credits")
                        subtitle: qsTr("Transcription and translation usage")
                    }

                    Rectangle {
                        Layout.fillWidth: true
                        Layout.preferredHeight: creditsContent.implicitHeight + Theme.spacingMd * 2
                        radius: Theme.panelRadius
                        color: Theme.surfaceRaised

                        ColumnLayout {
                            id: creditsContent
                            anchors.fill: parent
                            anchors.margins: Theme.spacingMd
                            spacing: Theme.spacingSm

                            // Loading state
                            RowLayout {
                                visible: licenseScrollView.licenseManager ? licenseScrollView.licenseManager.creditsLoading : false
                                Layout.fillWidth: true
                                spacing: Theme.spacingSm

                                BusyIndicator {
                                    implicitWidth: 16
                                    implicitHeight: 16
                                    running: true
                                }

                                Label {
                                    text: qsTr("Loading...")
                                    color: Theme.textSecondary
                                    font.pixelSize: 12
                                }
                            }

                            // Error state
                            Label {
                                visible: licenseScrollView.licenseManager &&
                                         !licenseScrollView.licenseManager.creditsLoading &&
                                         licenseScrollView.licenseManager.creditsError.length > 0
                                text: licenseScrollView.licenseManager ? licenseScrollView.licenseManager.creditsError : ""
                                color: Theme.statusDanger
                                font.pixelSize: 12
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }

                            // Credits display (loaded successfully)
                            ColumnLayout {
                                visible: licenseScrollView.licenseManager &&
                                         !licenseScrollView.licenseManager.creditsLoading &&
                                         licenseScrollView.licenseManager.creditsError.length === 0
                                Layout.fillWidth: true
                                spacing: Theme.spacingSm

                                RowLayout {
                                    Layout.fillWidth: true

                                    Label {
                                        readonly property int remaining: licenseScrollView.licenseManager ?
                                            (licenseScrollView.licenseManager.creditsLimit - licenseScrollView.licenseManager.creditsUsed) : 0
                                        text: qsTr("%1 / %2 credits remaining")
                                            .arg(remaining.toLocaleString())
                                            .arg(licenseScrollView.licenseManager ?
                                                 licenseScrollView.licenseManager.creditsLimit.toLocaleString() : "0")
                                        font.pixelSize: 14
                                        color: Theme.textPrimary
                                    }

                                    Item { Layout.fillWidth: true }

                                    Label {
                                        text: qsTr("%1% remaining").arg(
                                            licenseScrollView.licenseManager ?
                                            (100 - licenseScrollView.licenseManager.creditsPercentage).toFixed(1) : "100.0"
                                        )
                                        font.pixelSize: 12
                                        color: Theme.textSecondary
                                    }
                                }

                                ProgressBar {
                                    id: creditsProgressBar
                                    Layout.fillWidth: true
                                    Layout.preferredHeight: 8
                                    from: 0
                                    to: 100
                                    value: licenseScrollView.licenseManager ?
                                           (100 - licenseScrollView.licenseManager.creditsPercentage) : 100

                                    background: Rectangle {
                                        implicitWidth: 200
                                        implicitHeight: 8
                                        radius: 4
                                        color: Theme.surface
                                        border.color: Theme.surfaceBorder
                                        border.width: 1
                                    }

                                    contentItem: Item {
                                        implicitWidth: 200
                                        implicitHeight: 8

                                        Rectangle {
                                            width: creditsProgressBar.visualPosition * parent.width
                                            height: parent.height
                                            radius: 4
                                            color: Theme.accent
                                        }
                                    }
                                }

                                RowLayout {
                                    Layout.fillWidth: true
                                    Layout.topMargin: Theme.spacingSm
                                    spacing: Theme.spacingMd

                                    Controls.SecondaryButton {
                                        text: licenseScrollView.licenseManager && licenseScrollView.licenseManager.buyCreditsLoading
                                            ? qsTr("Opening…")
                                            : qsTr("Buy Credits")
                                        enabled: licenseScrollView.licenseManager && !licenseScrollView.licenseManager.buyCreditsLoading
                                        onClicked: {
                                            if (licenseScrollView.licenseManager) {
                                                licenseScrollView.licenseManager.buyCredits()
                                            }
                                        }
                                    }

                                    Controls.SecondaryButton {
                                        text: licenseScrollView.licenseManager && licenseScrollView.licenseManager.creditsLoading
                                            ? qsTr("Refreshing…")
                                            : qsTr("Refresh")
                                        enabled: licenseScrollView.licenseManager && !licenseScrollView.licenseManager.creditsLoading
                                        onClicked: {
                                            if (licenseScrollView.licenseManager) {
                                                licenseScrollView.licenseManager.refreshCredits()
                                            }
                                        }
                                    }

                                    Item { Layout.fillWidth: true }
                                }
                            }
                        }
                    }
                }

                // Activation form (only show if not activated)
                ColumnLayout {
                    visible: !licenseState.isValid
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    Forms.LabeledField {
                        label: qsTr("License key")
                        helperText: qsTr("Enter the license key you received after purchase.")

                        ColumnLayout {
                            Layout.fillWidth: true
                            spacing: Theme.spacingSm

                            TextField {
                                id: licenseKeyField
                                Layout.fillWidth: true
                                placeholderText: qsTr("XXXX-XXXX-XXXX-XXXX")
                                selectByMouse: true
                                enabled: !licenseState.isActivating

                                onTextChanged: {
                                    licenseState.errorMessage = ""
                                    licenseState.successMessage = ""
                                }
                            }

                            RowLayout {
                                Layout.fillWidth: true
                                spacing: Theme.spacingMd

                                Controls.SecondaryButton {
                                    text: licenseState.isActivating ? qsTr("Activating...") : qsTr("Activate")
                                    enabled: !licenseState.isActivating && licenseKeyField.text.trim().length > 0

                                    onClicked: {
                                        licenseState.isActivating = true
                                        licenseState.errorMessage = ""
                                        licenseState.successMessage = ""
                                        licenseScrollView.licenseManager.activateLicense(licenseKeyField.text.trim())
                                    }
                                }

                                Item { Layout.fillWidth: true }
                            }
                        }
                    }

                    // Error message
                    Label {
                        visible: licenseState.errorMessage.length > 0
                        text: licenseState.errorMessage
                        color: Theme.statusDanger
                        font.pixelSize: 12
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }

                    // Success message
                    Label {
                        visible: licenseState.successMessage.length > 0
                        text: licenseState.successMessage
                        color: "#4CAF50"
                        font.pixelSize: 12
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }

                    Controls.SecondaryButton {
                        text: qsTr("View Plans & Pricing")
                        onClicked: {
                            var lang = root.languageController ? root.languageController.currentLanguage : "en"
                            Qt.openUrlExternally(licenseScrollView.licenseManager.localizedPricingUrl(lang))
                        }
                    }
                }

                // Deactivate section (only show if activated)
                ColumnLayout {
                    visible: licenseState.isValid
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    Layouts.SectionHeader {
                        title: qsTr("Manage License")
                        subtitle: qsTr("Remove the license from this device")
                    }

                    Label {
                        text: qsTr("Removing the license will disable cloud transcription on this device. You can re-activate it later with the same license key.")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Remove License")
                        onClicked: {
                            licenseScrollView.licenseManager.clearLicense()
                            licenseState.isValid = false
                            licenseState.successMessage = ""
                            licenseState.errorMessage = ""
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: exportPage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                QtObject {
                    id: exportPreferencesState
                    property var values: root.settingsModel.export_preferences()
                }

                Connections {
                    target: root.settingsModel
                    function onExport_preferences_changed(values) {
                        exportPreferencesState.values = values
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Video")
                    subtitle: qsTr("Defaults used when opening the export panel")
                }

                Forms.LabeledField {
                    label: qsTr("Video encoder")
                    helperText: qsTr("Choose the codec we target when a full recode is required.")

                    ComboBox {
                        id: encoderCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: root.videoEncoderOptions

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: setCurrentValue(exportPreferencesState.values.video_encoder_override)

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) {
                                return
                            }
                            exportPreferencesState.values = root.settingsModel.set_export_preference("video_encoder_override", option.value)
                        }

                        Connections {
                            target: root.settingsModel
                            function onExport_preferences_changed(values) {
                                encoderCombo.setCurrentValue(values.video_encoder_override)
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Quality preset")
                    helperText: qsTr("CRF targets for re-encoded exports. Higher values create larger files.")

                    ComboBox {
                        id: qualityCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: root.videoQualityOptions

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 1
                        }

                        Component.onCompleted: setCurrentValue(exportPreferencesState.values.video_quality_index)

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) {
                                return
                            }
                            exportPreferencesState.values = root.settingsModel.set_export_preference("video_quality_index", option.value)
                        }

                        Connections {
                            target: root.settingsModel
                            function onExport_preferences_changed(values) {
                                qualityCombo.setCurrentValue(values.video_quality_index)
                            }
                        }
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Audio & timeline")
                    subtitle: qsTr("Control the initial state of export toggles")
                }

                RowLayout {
                    Layout.fillWidth: true

                    Label {
                        text: qsTr("Export individual segments by default")
                        Layout.fillWidth: true
                    }

                    Switch {
                        id: segmentSwitch
                        checked: !!exportPreferencesState.values.segment_mode
                        onToggled: exportPreferencesState.values = root.settingsModel.set_export_preference("segment_mode", checked)

                        Connections {
                            target: root.settingsModel
                            function onExport_preferences_changed(values) {
                                if (segmentSwitch.checked !== !!values.segment_mode) {
                                    segmentSwitch.checked = !!values.segment_mode
                                }
                            }
                        }
                    }
                }

                RowLayout {
                    Layout.fillWidth: true

                    Label {
                        text: qsTr("Include mixdown track")
                        Layout.fillWidth: true
                    }

                    Switch {
                        id: mixSwitch
                        checked: !!exportPreferencesState.values.mix_enabled
                        onToggled: exportPreferencesState.values = root.settingsModel.set_export_preference("mix_enabled", checked)

                        Connections {
                            target: root.settingsModel
                            function onExport_preferences_changed(values) {
                                if (mixSwitch.checked !== !!values.mix_enabled) {
                                    mixSwitch.checked = !!values.mix_enabled
                                }
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Timeline export format")
                    helperText: qsTr("Used when generating XML/OTIO sidecars.")

                    ComboBox {
                        id: xmlCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: root.xmlModeOptions

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0
                        }

                        Component.onCompleted: setCurrentValue(exportPreferencesState.values.xml_mode)

                        onActivated: function(index) {
                            const option = model[index]
                            if (!option) {
                                return
                            }
                            exportPreferencesState.values = root.settingsModel.set_export_preference("xml_mode", option.value)
                        }

                        Connections {
                            target: root.settingsModel
                            function onExport_preferences_changed(values) {
                                xmlCombo.setCurrentValue(values.xml_mode)
                            }
                        }
                    }
                }

                Controls.SecondaryButton {
                    text: qsTr("Restore export preferences")
                    Layout.alignment: Qt.AlignLeft
                    onClicked: exportPreferencesState.values = root.settingsModel.reset_export_preferences()
                }

                Layouts.SectionHeader {
                    title: qsTr("Subtitles")
                    subtitle: qsTr("Formatting options for exported subtitles")
                }

                Forms.LabeledField {
                    label: qsTr("Characters per line")
                    helperText: qsTr("Maximum characters per subtitle line. Standard is 35-50.")

                    TextField {
                        id: subtitleCharsField
                        Layout.preferredWidth: 80
                        text: root.settingsModel ? root.settingsModel.subtitle_max_chars_per_line().toString() : "50"
                        validator: IntValidator { bottom: 20; top: 100 }
                        selectByMouse: true
                        onEditingFinished: {
                            if (!root.settingsModel) return
                            var val = parseInt(text)
                            if (!isNaN(val)) {
                                root.settingsModel.set_subtitle_max_chars_per_line(val)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onSubtitle_settings_changed() {
                                subtitleCharsField.text = root.settingsModel.subtitle_max_chars_per_line().toString()
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Maximum lines")
                    helperText: qsTr("Number of lines per subtitle. Industry standard is 2.")

                    ComboBox {
                        id: subtitleLinesCombo
                        Layout.fillWidth: true
                        textRole: "label"
                        valueRole: "value"
                        model: [
                            { label: qsTr("1 line"), value: 1 },
                            { label: qsTr("2 lines"), value: 2 },
                            { label: qsTr("3 lines"), value: 3 }
                        ]

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].value === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 1  // Default to 2 lines
                        }

                        Component.onCompleted: {
                            if (root.settingsModel) {
                                setCurrentValue(root.settingsModel.subtitle_max_lines())
                            }
                        }

                        onActivated: function(index) {
                            const option = model[index]
                            if (option && root.settingsModel) {
                                root.settingsModel.set_subtitle_max_lines(option.value)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onSubtitle_settings_changed() {
                                subtitleLinesCombo.setCurrentValue(root.settingsModel.subtitle_max_lines())
                            }
                        }
                    }
                }

                Forms.LabeledField {
                    label: qsTr("Maximum duration")
                    helperText: qsTr("Maximum seconds per subtitle. Longer sentences split into multiple entries.")

                    TextField {
                        id: subtitleDurationField
                        Layout.preferredWidth: 80
                        text: root.settingsModel ? root.settingsModel.subtitle_max_duration().toFixed(1) : "7.0"
                        validator: DoubleValidator { bottom: 2.0; top: 30.0; decimals: 1 }
                        selectByMouse: true
                        onEditingFinished: {
                            if (!root.settingsModel) return
                            var val = parseFloat(text)
                            if (!isNaN(val)) {
                                root.settingsModel.set_subtitle_max_duration(val)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onSubtitle_settings_changed() {
                                subtitleDurationField.text = root.settingsModel.subtitle_max_duration().toFixed(1)
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: performancePage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                Layouts.SectionHeader {
                    title: qsTr("Hardware Acceleration")
                    subtitle: qsTr("GPU and decoding settings")
                }

                Forms.LabeledField {
                    label: qsTr("Hardware decoding")
                    helperText: qsTr("Use GPU for video decoding. May improve performance with high-resolution video.")

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingMd

                        ComboBox {
                            id: hwDecodingCombo
                            Layout.preferredWidth: 240
                            textRole: "label"
                            valueRole: "value"
                            model: root.settingsModel ? root.settingsModel.available_hardware_devices() : []

                            function setCurrentValue(value) {
                                for (let i = 0; i < model.length; ++i) {
                                    if (model[i].value === value) {
                                        currentIndex = i
                                        return
                                    }
                                }
                                currentIndex = 0
                            }

                            Component.onCompleted: {
                                if (root.settingsModel) {
                                    setCurrentValue(root.settingsModel.hardware_decoding())
                                }
                            }

                            onActivated: function(index) {
                                const option = model[index]
                                if (option && root.settingsModel) {
                                    root.settingsModel.set_hardware_decoding(option.value)
                                }
                            }

                            Connections {
                                target: root.settingsModel || null
                                ignoreUnknownSignals: true
                                function onHardware_decoding_changed(value) {
                                    hwDecodingCombo.setCurrentValue(value)
                                }
                            }
                        }

                        Label {
                            id: hwStatusLabel
                            readonly property bool hasMedia: root.appWorkspaceController ? root.appWorkspaceController.mediaStatus === "ready" : false
                            readonly property bool hwActive: root.appWorkspaceController ? root.appWorkspaceController.hwaccelActive : false
                            readonly property string hwDevice: root.appWorkspaceController ? root.appWorkspaceController.hwaccelDevice : ""
                            readonly property bool showHwDevice: hwActive && hwDevice.length > 0

                            visible: hasMedia
                            text: showHwDevice
                                ? qsTr("Currently using: %1").arg(hwDevice)
                                : qsTr("Currently using: Software decoding")
                            font.pixelSize: 12
                            color: showHwDevice ? Theme.statusSuccess : Theme.textSecondary
                        }
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Transcription")
                    subtitle: qsTr("Local Whisper processing")
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Try to use GPU for transcribing")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Uses GPU acceleration for local Whisper transcription when available")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Switch {
                        id: transcriptionGpuSwitch
                        checked: root.settingsModel ? root.settingsModel.transcription_use_gpu() : false
                        onToggled: {
                            if (root.settingsModel) {
                                root.settingsModel.set_transcription_use_gpu(checked)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onTranscription_use_gpu_changed(enabled) {
                                if (transcriptionGpuSwitch.checked !== enabled) {
                                    transcriptionGpuSwitch.checked = enabled
                                }
                            }
                        }
                    }
                }

                // GPU device picker - shown when GPU mode is enabled and multiple devices available
                RowLayout {
                    visible: transcriptionGpuSwitch.checked && root.transcriptionController && root.transcriptionController.availableGpuDevices.length > 1
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("GPU device")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Select which GPU to use for transcription")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    ComboBox {
                        id: gpuDeviceCombo
                        Layout.preferredWidth: 200
                        textRole: "name"
                        valueRole: "index"
                        model: root.transcriptionController ? root.transcriptionController.availableGpuDevices : []

                        function setCurrentValue(value) {
                            for (let i = 0; i < model.length; ++i) {
                                if (model[i].index === value) {
                                    currentIndex = i
                                    return
                                }
                            }
                            currentIndex = 0  // Fallback to first device
                        }

                        Component.onCompleted: {
                            if (root.settingsModel) {
                                setCurrentValue(root.settingsModel.transcription_gpu_device())
                            }
                        }

                        onActivated: function(index) {
                            const option = model[index]
                            if (option && root.settingsModel) {
                                root.settingsModel.set_transcription_gpu_device(option.index)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onTranscription_gpu_device_changed(device) {
                                gpuDeviceCombo.setCurrentValue(device)
                            }
                        }
                    }
                }

                // GPU availability warning
                Label {
                    visible: transcriptionGpuSwitch.checked && root.transcriptionController && !root.transcriptionController.gpuBackendAvailable
                    text: qsTr("No GPU available - transcription will use CPU")
                    font.pixelSize: 12
                    color: Theme.statusDanger
                    wrapMode: Text.WordWrap
                    Layout.fillWidth: true
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Enable tinydiarize speaker turns (experimental)")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Only supported with the English > Medium-size model; needs a separate model download")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Switch {
                        id: transcriptionTdrzSwitch
                        checked: root.settingsModel ? root.settingsModel.transcription_use_tinydiarize() : false
                        onToggled: {
                            if (root.settingsModel) {
                                root.settingsModel.set_transcription_use_tinydiarize(checked)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onTranscription_use_tinydiarize_changed(enabled) {
                                if (transcriptionTdrzSwitch.checked !== enabled) {
                                    transcriptionTdrzSwitch.checked = enabled
                                }
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }

    Component {
        id: updatesPage

        ScrollView {
            clip: true
            Layout.fillWidth: true
            Layout.fillHeight: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                Layouts.SectionHeader {
                    title: qsTr("Update checks")
                    subtitle: qsTr("Control automatic update notifications")
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Check for updates on startup")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("We'll notify you when a new version is available")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Switch {
                        id: updateCheckSwitch
                        checked: root.settingsModel ? (root.settingsModel.check_updates_on_startup() === true) : false
                        onToggled: {
                            if (root.settingsModel) {
                                root.settingsModel.set_check_updates_on_startup(checked)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onCheck_updates_on_startup_changed(enabled) {
                                if (updateCheckSwitch.checked !== enabled) {
                                    updateCheckSwitch.checked = enabled
                                }
                            }
                        }
                    }
                }

                Controls.SecondaryButton {
                    text: root.updateController && root.updateController.isChecking ? qsTr("Checking...") : qsTr("Check now")
                    enabled: root.updateController && !root.updateController.isChecking
                    onClicked: {
                        if (root.updateController) {
                            root.updateController.checkForUpdates(true)  // manual=true to show popup immediately
                        }
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Privacy")
                    subtitle: qsTr("Data collection preferences")
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Allow anonymous usage data")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Help us improve the app by sharing anonymous analytics")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Switch {
                        id: telemetrySwitch
                        checked: root.settingsModel ? (root.settingsModel.allow_telemetry() === true) : false
                        onToggled: {
                            if (root.settingsModel) {
                                root.settingsModel.set_allow_telemetry(checked)
                            }
                        }

                        Connections {
                            target: root.settingsModel || null
                            ignoreUnknownSignals: true
                            function onAllow_telemetry_changed(enabled) {
                                if (telemetrySwitch.checked !== enabled) {
                                    telemetrySwitch.checked = enabled
                                }
                            }
                        }
                    }
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Privacy policy")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Learn how we handle your data")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Controls.SecondaryButton {
                        text: qsTr("View")
                        onClicked: Qt.openUrlExternally(root.settingsModel.api_base_url() + "/privacy/")
                    }
                }

                Layouts.SectionHeader {
                    title: qsTr("Community")
                    subtitle: qsTr("Get help and share feedback")
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Join our Discord")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Get help, report bugs, and suggest features")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    ToolButton {
                        implicitWidth: 36
                        implicitHeight: 36
                        icon.source: Theme.iconUrl("brand-discord")
                        icon.width: 24
                        icon.height: 24
                        icon.color: Theme.textPrimary
                        onClicked: Qt.openUrlExternally("https://discord.gg/uYGkyfzU4c")
                    }
                }

                RowLayout {
                    Layout.fillWidth: true
                    spacing: Theme.spacingMd

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 2

                        Label {
                            text: qsTr("Send feedback")
                            font.pixelSize: 14
                            color: Theme.textPrimary
                        }

                        Label {
                            text: qsTr("Report issues or suggest improvements")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }
                    }

                    Controls.SecondaryButton {
                        text: qsTr("Send")
                        onClicked: {
                            root.close()
                            root.openFeedback()
                        }
                    }
                }

                Item { Layout.fillHeight: true }
            }
        }
    }
}
