import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import components.layout 1.0
import components.media 1.0
import components.workspace 1.0
import features.transcription 1.0
import features.transcription.panels 1.0
import SMC.Controllers 1.0
import styles 1.0

Item {
    id: root
    objectName: "TranscriptionWorkspaceTab"
    width: 1280
    height: 720
    implicitWidth: 1280
    implicitHeight: 720

    required property QmlTranscriptionController controller
    required property var appController
    readonly property var workspaceController: appController.cutWorkspaceController
    required property var timelineBackend
    required property var settingsModel
    required property var languageController
    required property var toastManager
    signal feedbackRequested()
    readonly property bool transcriptionBusy: controller.busy
    readonly property string transcriptionStatusText: controller.busy
                                                           ? qsTr("Transcribing…")
                                                           : controller.noSpeechDetected
                                                                 ? qsTr("No speech detected")
                                                                 : controller.transcriptSegments && controller.transcriptSegments.length > 0
                                                                       ? qsTr("%1 segments ready").arg(controller.transcriptSegments.length)
                                                                       : qsTr("Waiting for transcription")
    readonly property bool timelineReady: appController.mediaStatus === "ready"
    readonly property string timelineEmptyText: appController.mediaStatus === "loading"
                                                    ? qsTr("Building timeline…")
                                                    : appController.mediaStatus === "error"
                                                          ? appController.mediaError || qsTr("Timeline unavailable")
                                                          : qsTr("Open a media file to begin")
    readonly property real mediaDuration: appController.mediaDuration
    property bool syncTimelineToSelection: settingsModel ? settingsModel.transcript_sync_timeline() : true
    property bool tinydiarizeEnabled: settingsModel ? settingsModel.transcription_use_tinydiarize() : false
    readonly property bool speakerTurnsToggleVisible: {
        if (!controller) return false
        if (controller.transcriptionBackend === "cloud") return true
        return root.tinydiarizeEnabled && controller.modelSize === "small.en"
    }

    WorkspaceViewPreferences {
        id: viewPrefs
        settingsModel: root.settingsModel
        workspaceKey: "transcribe"
    }

    Connections {
        target: root.settingsModel || null
        ignoreUnknownSignals: true
        function onTranscription_use_tinydiarize_changed(enabled) {
            root.tinydiarizeEnabled = enabled
        }
    }

    function formatTime(value) {
        const totalMs = Math.max(0, Math.round(Number(value || 0) * 1000))
        const hours = Math.floor(totalMs / 3600000)
        const minutes = Math.floor((totalMs % 3600000) / 60000)
        const seconds = Math.floor((totalMs % 60000) / 1000)
        const centiseconds = Math.floor((totalMs % 1000) / 10)
        const hh = hours.toString().padStart(2, "0")
        const mm = minutes.toString().padStart(2, "0")
        const ss = seconds.toString().padStart(2, "0")
        const cs = centiseconds.toString().padStart(2, "0")
        if (hours > 0) {
            return hh + ":" + mm + ":" + ss + "." + cs
        }
        return mm + ":" + ss + "." + cs
    }

    RowLayout {
        anchors.fill: parent
        spacing: Theme.spacingLg

        // Left Sidebar
        TranscriptionSidebar {
            id: sidebar
            Layout.preferredWidth: 320
            Layout.fillHeight: true
            controller: root.controller
            licenseManager: root.appController.licenseManager
            languageController: root.languageController
            settingsModel: root.settingsModel
            toastManager: root.toastManager

            onSearchRequested: transcriptView.openSearch()
            onFeedbackRequested: root.feedbackRequested()
        }

        // Main Content Area
        SplitView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            orientation: Qt.Vertical

            handle: Rectangle {
                implicitWidth: parent.width
                implicitHeight: Theme.spacingSm
                color: "transparent"

                // Drag handle indicator
                Rectangle {
                    anchors.centerIn: parent
                    width: 40
                    height: 3
                    radius: 1.5
                    color: Theme.surfaceBorder
                    opacity: SplitHandle.hovered || SplitHandle.pressed ? 1.0 : 0.5

                    Behavior on opacity { NumberAnimation { duration: 100 } }
                }
            }

            // Top: Preview & Timeline
            ColumnLayout {
                SplitView.preferredHeight: 400
                SplitView.fillWidth: true
                SplitView.minimumHeight: 200
                spacing: Theme.spacingLg

                MediaPreviewCard {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    appController: root.appController
                    workspaceController: root.workspaceController
                    showFrameStrip: false
                }

                TimelineSection {
                    id: timelineSection
                    Layout.fillWidth: true
                    Layout.bottomMargin: Theme.spacingLg
                    timelineBackend: root.timelineBackend
                    appController: root.appController
                    showOverviewTimeline: viewPrefs.prefs.showOverviewTimeline
                    showAudioTimeline: viewPrefs.prefs.showAudioTimeline
                    timelineReady: root.timelineReady
                    emptyStateText: root.timelineEmptyText
                }
            }

            // Bottom: Transcript Editor
            Card {
                SplitView.fillWidth: true
                SplitView.fillHeight: true
                SplitView.minimumHeight: 150
                contentPadding: 0

                ColumnLayout {
                    anchors.fill: parent
                    spacing: 0

                    // Custom header with timestamp toggle
                    Item {
                        Layout.fillWidth: true
                        Layout.preferredHeight: 48

                        RowLayout {
                            anchors.fill: parent
                            anchors.leftMargin: Theme.spacingLg
                            anchors.rightMargin: Theme.spacingMd
                            spacing: Theme.spacingMd

                            Label {
                                text: qsTr("Transcript")
                                color: Theme.textPrimary
                                font.pixelSize: 16
                                font.weight: Font.DemiBold
                            }

                            // Tag-style toggle button
                            Rectangle {
                                Layout.preferredWidth: timestampTagLabel.implicitWidth + Theme.spacingMd * 2
                                Layout.preferredHeight: timestampTagLabel.implicitHeight + Theme.spacingXs * 2
                                radius: height / 2
                                color: transcriptView.showTimestamps ? Theme.surfaceRaised : Theme.surface
                                border.width: 1
                                border.color: transcriptView.showTimestamps ? Theme.surfaceBorder : "transparent"

                                Label {
                                    id: timestampTagLabel
                                    anchors.centerIn: parent
                                    text: qsTr("Show timestamps")
                                    font.pixelSize: 12
                                    color: transcriptView.showTimestamps ? Theme.textPrimary : Theme.textSecondary
                                }

                                MouseArea {
                                    anchors.fill: parent
                                    cursorShape: Qt.PointingHandCursor
                                    hoverEnabled: true
                                    onClicked: root.settingsModel.set_transcript_show_timestamps(!transcriptView.showTimestamps)
                                    ToolTip.visible: containsMouse
                                    ToolTip.delay: 500
                                    ToolTip.text: qsTr("Display timestamps in the left margin of the transcript.")
                                }
                            }

                            // Sync timeline toggle
                            Rectangle {
                                Layout.preferredWidth: syncTagLabel.implicitWidth + Theme.spacingMd * 2
                                Layout.preferredHeight: syncTagLabel.implicitHeight + Theme.spacingXs * 2
                                radius: height / 2
                                color: root.syncTimelineToSelection ? Theme.surfaceRaised : Theme.surface
                                border.width: 1
                                border.color: root.syncTimelineToSelection ? Theme.surfaceBorder : "transparent"

                                Label {
                                    id: syncTagLabel
                                    anchors.centerIn: parent
                                    text: qsTr("Sync timeline to selection")
                                    font.pixelSize: 12
                                    color: root.syncTimelineToSelection ? Theme.textPrimary : Theme.textSecondary
                                }

                                MouseArea {
                                    anchors.fill: parent
                                    cursorShape: Qt.PointingHandCursor
                                    hoverEnabled: true
                                    onClicked: root.settingsModel.set_transcript_sync_timeline(!root.syncTimelineToSelection)
                                    ToolTip.visible: containsMouse
                                    ToolTip.delay: 500
                                    ToolTip.text: qsTr("Automatically center the audio timeline on selected text.")
                                }
                            }

                            // Show popup toggle
                            Rectangle {
                                objectName: "transcriptPopupToggle"
                                Layout.preferredWidth: popupTagLabel.implicitWidth + Theme.spacingMd * 2
                                Layout.preferredHeight: popupTagLabel.implicitHeight + Theme.spacingXs * 2
                                radius: height / 2
                                color: transcriptView.showPopup ? Theme.surfaceRaised : Theme.surface
                                border.width: 1
                                border.color: transcriptView.showPopup ? Theme.surfaceBorder : "transparent"

                                Label {
                                    id: popupTagLabel
                                    anchors.centerIn: parent
                                    text: qsTr("Show controls popup")
                                    font.pixelSize: 12
                                    color: transcriptView.showPopup ? Theme.textPrimary : Theme.textSecondary
                                }

                                MouseArea {
                                    anchors.fill: parent
                                    cursorShape: Qt.PointingHandCursor
                                    hoverEnabled: true
                                    onClicked: {
                                        root.settingsModel.set_transcript_show_popup(!transcriptView.showPopup)
                                        root.settingsModel.record_milestone("transcript_popup_toggle")
                                    }
                                    ToolTip.visible: containsMouse
                                    ToolTip.delay: 500
                                    ToolTip.text: qsTr("Show a floating toolbar with playback and cut controls when selecting text.")
                                }
                            }

                            // Select pauses toggle
                            Rectangle {
                                Layout.preferredWidth: pausesTagLabel.implicitWidth + Theme.spacingMd * 2
                                Layout.preferredHeight: pausesTagLabel.implicitHeight + Theme.spacingXs * 2
                                radius: height / 2
                                color: transcriptView.selectPauses ? Theme.surfaceRaised : Theme.surface
                                border.width: 1
                                border.color: transcriptView.selectPauses ? Theme.surfaceBorder : "transparent"

                                Label {
                                    id: pausesTagLabel
                                    anchors.centerIn: parent
                                    text: qsTr("Select pauses between words")
                                    font.pixelSize: 12
                                    color: transcriptView.selectPauses ? Theme.textPrimary : Theme.textSecondary
                                }

                                MouseArea {
                                    anchors.fill: parent
                                    cursorShape: Qt.PointingHandCursor
                                    hoverEnabled: true
                                    onClicked: root.settingsModel.set_transcript_select_pauses(!transcriptView.selectPauses)
                                    ToolTip.visible: containsMouse
                                    ToolTip.delay: 500
                                    ToolTip.text: qsTr("Extend selection to include silent gaps before and after words. Prevents leftover pauses when cutting consecutive words separately.")
                                }
                            }

                            // Show speaker turns toggle
                            Rectangle {
                                visible: root.speakerTurnsToggleVisible
                                opacity: root.transcriptionBusy ? 0.6 : 1.0
                                Layout.preferredWidth: speakerTurnsTagLabel.implicitWidth + Theme.spacingMd * 2
                                Layout.preferredHeight: speakerTurnsTagLabel.implicitHeight + Theme.spacingXs * 2
                                radius: height / 2
                                color: transcriptView.showSpeakerTurns ? Theme.surfaceRaised : Theme.surface
                                border.width: 1
                                border.color: transcriptView.showSpeakerTurns ? Theme.surfaceBorder : "transparent"

                                Label {
                                    id: speakerTurnsTagLabel
                                    anchors.centerIn: parent
                                    text: qsTr("Show speaker turns")
                                    font.pixelSize: 12
                                    color: transcriptView.showSpeakerTurns ? Theme.textPrimary : Theme.textSecondary
                                }

                                MouseArea {
                                    anchors.fill: parent
                                    cursorShape: root.transcriptionBusy ? Qt.ForbiddenCursor : Qt.PointingHandCursor
                                    hoverEnabled: true
                                    enabled: !root.transcriptionBusy
                                    onClicked: root.settingsModel.set_transcript_show_speaker_turns(!transcriptView.showSpeakerTurns)
                                    ToolTip.visible: containsMouse
                                    ToolTip.delay: 500
                                    ToolTip.text: qsTr("Show speaker turn markers in the transcript when available.")
                                }
                            }

                            Item { Layout.fillWidth: true }
                        }

                        // Bottom border
                        Rectangle {
                            anchors.left: parent.left
                            anchors.right: parent.right
                            anchors.bottom: parent.bottom
                            height: 1
                            color: Theme.surfaceBorder
                        }
                    }

                    // Transcript Surface
                    TranscriptSurface {
                        id: transcriptView
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        controller: root.controller
                        appController: root.appController
                        settingsModel: root.settingsModel
                    }

                    // Center audio timeline on transcript selection
                    Connections {
                        target: transcriptView
                        function onSelectionTimeStartChanged() {
                            if (transcriptView.selectionActive && viewPrefs.prefs.showAudioTimeline && root.syncTimelineToSelection) {
                                timelineSection.centerOnRange(
                                    transcriptView.selectionTimeStart,
                                    transcriptView.selectionTimeEnd
                                )
                            }
                        }
                        function onSelectionTimeEndChanged() {
                            if (transcriptView.selectionActive && viewPrefs.prefs.showAudioTimeline && root.syncTimelineToSelection) {
                                timelineSection.centerOnRange(
                                    transcriptView.selectionTimeStart,
                                    transcriptView.selectionTimeEnd
                                )
                            }
                        }
                    }
                }
            }
        }
    }

    // Keep sync setting in sync with settingsModel
    Connections {
        target: root.settingsModel
        function onTranscript_sync_timeline_changed(enabled) {
            root.syncTimelineToSelection = enabled
        }
    }

    Component.onCompleted: {
        if (controller)
            controller.refresh_tracks()
    }

    // Ctrl+F to open transcript search
    Shortcut {
        sequence: "Ctrl+F"
        context: Qt.WindowShortcut
        enabled: root.visible && sidebar.hasTranscript
        onActivated: transcriptView.openSearch()
    }
}
