import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

Rectangle {
    id: root

    property alias searchText: searchField.text
    property int matchCount: 0
    property int currentMatchIndex: -1

    // Search options
    property alias highlightAll: highlightAllCheck.checked
    property alias matchCase: matchCaseCheck.checked
    property alias wholeWords: wholeWordsCheck.checked
    property alias includeCuts: includeCutsCheck.checked

    signal searchChanged(string text)
    signal searchOptionsChanged()  // matchCase, wholeWords - requires re-search
    signal highlightOptionsChanged()  // highlightAll - visual only
    signal nextMatch()
    signal prevMatch()
    signal closed()

    implicitHeight: 40
    color: Theme.surfaceRaised
    border.color: Theme.surfaceBorder
    border.width: 1
    radius: Theme.panelRadius - 4

    RowLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingSm
        spacing: Theme.spacingSm

        TextField {
            id: searchField
            Layout.fillWidth: true
            placeholderText: qsTr("Search transcript...")
            selectByMouse: true
            font.pixelSize: 13

            background: Rectangle {
                color: Theme.surface
                radius: Theme.panelRadius - 4
                border.color: searchField.activeFocus ? Theme.accent : Theme.surfaceBorder
                border.width: 1
            }

            onTextChanged: root.searchChanged(text)

            Keys.onReturnPressed: root.nextMatch()
            Keys.onEnterPressed: root.nextMatch()
            Keys.onEscapePressed: root.closed()
            Keys.onUpPressed: root.prevMatch()
            Keys.onDownPressed: root.nextMatch()
        }

        // Match count label
        Label {
            text: root.matchCount > 0
                ? qsTr("%1/%2").arg(root.currentMatchIndex + 1).arg(root.matchCount)
                : root.searchText.length > 0 ? qsTr("0/0") : ""
            color: root.matchCount > 0 ? Theme.textSecondary : Theme.statusDanger
            font.pixelSize: 12
            visible: root.searchText.length > 0
            Layout.preferredWidth: 50
            horizontalAlignment: Text.AlignHCenter
        }

        // Previous match button
        ToolButton {
            id: prevBtn
            Layout.preferredWidth: 28
            Layout.preferredHeight: 28
            enabled: root.matchCount > 0
            onClicked: root.prevMatch()

            contentItem: Label {
                text: "▲"
                color: prevBtn.enabled ? Theme.textPrimary : Theme.textSecondary
                font.pixelSize: 12
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
            }

            background: Rectangle {
                radius: 4
                color: prevBtn.pressed
                    ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.12)
                    : prevBtn.hovered
                        ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.06)
                        : "transparent"
            }
        }

        // Next match button
        ToolButton {
            id: nextBtn
            Layout.preferredWidth: 28
            Layout.preferredHeight: 28
            enabled: root.matchCount > 0
            onClicked: root.nextMatch()

            contentItem: Label {
                text: "▼"
                color: nextBtn.enabled ? Theme.textPrimary : Theme.textSecondary
                font.pixelSize: 12
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
            }

            background: Rectangle {
                radius: 4
                color: nextBtn.pressed
                    ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.12)
                    : nextBtn.hovered
                        ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.06)
                        : "transparent"
            }
        }

        // Separator
        Rectangle {
            Layout.preferredWidth: 1
            Layout.preferredHeight: 20
            color: Theme.surfaceBorder
        }

        // Highlight All checkbox
        CheckBox {
            id: highlightAllCheck
            text: qsTr("Highlight All")
            checked: false
            font.pixelSize: 11
            spacing: 4
            onToggled: root.highlightOptionsChanged()
        }

        // Match Case checkbox
        CheckBox {
            id: matchCaseCheck
            text: qsTr("Match Case")
            font.pixelSize: 11
            spacing: 4
            onToggled: root.searchOptionsChanged()
        }

        // Whole Words checkbox
        CheckBox {
            id: wholeWordsCheck
            text: qsTr("Whole Words")
            font.pixelSize: 11
            spacing: 4
            onToggled: root.searchOptionsChanged()
        }

        // Include Cuts checkbox
        CheckBox {
            id: includeCutsCheck
            text: qsTr("Include Cuts")
            checked: false
            font.pixelSize: 11
            spacing: 4
            onToggled: root.searchOptionsChanged()
        }

        // Close button
        ToolButton {
            id: closeBtn
            Layout.preferredWidth: 28
            Layout.preferredHeight: 28
            onClicked: root.closed()

            icon.source: Theme.iconUrl("x")
            icon.width: 16
            icon.height: 16
            icon.color: Theme.textSecondary

            background: Rectangle {
                radius: 4
                color: closeBtn.pressed
                    ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.12)
                    : closeBtn.hovered
                        ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.06)
                        : "transparent"
            }
        }
    }

    function focusInput() {
        searchField.forceActiveFocus()
        searchField.selectAll()
    }

    function clear() {
        searchField.text = ""
    }
}
