pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0
import features.transcription 1.0

Item {
    id: transcriptSurface
    required property var controller
    required property var appController
    required property var settingsModel

    // Font size from settings (with fallback)
    property int transcriptFontSize: transcriptSurface.settingsModel.transcript_font_size()
    readonly property bool controllerBusy: controller.busy
    readonly property bool hasTranscriptSegments: controller.transcriptSegments && controller.transcriptSegments.length > 0
    readonly property bool isRTL: TranscriptionLanguages.isRTL(controller.language)
    // Derive from actual state, not status string
    readonly property bool isTranscribing: controllerBusy || hasTranscriptSegments
    property bool transcriptReady: false  // Set true after we init the document
    Layout.fillWidth: true
    Layout.fillHeight: true
    objectName: "transcriptSurface"

    // Word ranges for selection ↔ timeline mapping (extended incrementally)
    property var wordRanges: []
    property var segCharStart: ({})

    // Timestamps for left column markers (time, charPos pairs)
    property var timestamps: []
    property bool showTimestamps: settingsModel ? settingsModel.transcript_show_timestamps() : true
    property bool showPopup: settingsModel ? settingsModel.transcript_show_popup() : true
    property bool selectPauses: settingsModel ? settingsModel.transcript_select_pauses() : true
    property bool showSpeakerTurns: settingsModel ? settingsModel.transcript_show_speaker_turns() : true

    // Format seconds as MM:SS or H:MM:SS
    function formatTimestamp(seconds) {
        const s = Math.floor(seconds)
        if (s >= 3600) {
            const h = Math.floor(s / 3600)
            const m = Math.floor((s % 3600) / 60)
            const sec = s % 60
            return h + ":" + m.toString().padStart(2, '0') + ":" + sec.toString().padStart(2, '0')
        }
        return Math.floor(s / 60).toString().padStart(2, '0') + ":" + (s % 60).toString().padStart(2, '0')
    }

    // Selection state
    property rect selectionRect: Qt.rect(0, 0, 0, 0)
    property bool selectionActive: false
    property real selectionTimeStart: 0
    property real selectionTimeEnd: 0
    property int selectionSegmentId: 0
    property int selectionAnchorRangeIdx: -1  // Tracks drag anchor word index for stable selection
    property int selectionRangeMinIdx: -1  // First word index in selection
    property int selectionRangeMaxIdx: -1  // Last word index in selection

    // Search state
    property bool searchActive: false
    property string searchText: ""
    property var searchMatches: []  // Array of {charStart, charEnd, rangeIdx, timeStart}
    property int currentMatchIndex: -1

    // Extended selection times (for "Cut extended" - extends to adjacent word boundaries)
    readonly property real selectionTimeStartExtended: {
        if (selectionRangeMinIdx <= 0 || wordRanges.length === 0) return selectionTimeStart
        return wordRanges[selectionRangeMinIdx - 1].t1
    }
    readonly property real selectionTimeEndExtended: {
        if (selectionRangeMaxIdx < 0 || selectionRangeMaxIdx >= wordRanges.length - 1) return selectionTimeEnd
        return wordRanges[selectionRangeMaxIdx + 1].t0
    }

    // Store selection char positions for recalculating rect on scroll
    property int selectionCharStart: 0
    property int selectionCharEnd: 0

    // Whether selection is visible in the viewport (for hiding popover when scrolled out)
    property bool selectionInView: true

    // Recalculate selection rect (called on selection change and scroll)
    function updateSelectionRect() {
        if (selectionCharStart === selectionCharEnd) return
        const rectStart = transcriptText.positionToRectangle(selectionCharStart)
        const rectEnd = transcriptText.positionToRectangle(selectionCharEnd)
        // Account for scroll offset - convert from content coords to viewport coords
        const scrollY = transcriptScroll.contentItem ? (transcriptScroll.contentItem as Flickable).contentY : 0
        const viewportHeight = transcriptScroll.height

        const contentTop = Math.min(rectStart.y, rectEnd.y)
        const contentBottom = Math.max(rectStart.y + rectStart.height, rectEnd.y + rectEnd.height)
        const viewportTop = contentTop - scrollY
        const viewportBottom = contentBottom - scrollY

        // Check if selection is visible in viewport
        selectionInView = viewportBottom > 0 && viewportTop < viewportHeight

        selectionRect = Qt.rect(
            Math.min(rectStart.x, rectEnd.x),
            viewportTop,
            Math.max(rectStart.x + rectStart.width, rectEnd.x + rectEnd.width) - Math.min(rectStart.x, rectEnd.x),
            contentBottom - contentTop
        )
    }

    function clearState() {
        wordRanges = []
        segCharStart = ({})
        timestamps = []
        selectionActive = false
        transcriptReady = false
        closeSearch()
    }

    // Clear ranges only (for content reload, keeps document ready)
    function clearRanges() {
        wordRanges = []
        segCharStart = ({})
        timestamps = []
        selectionActive = false
        closeSearch()
    }

    // Search functions
    function openSearch() {
        searchActive = true
        searchBar.focusInput()
    }

    function closeSearch() {
        searchActive = false
        searchText = ""
        searchMatches = []
        currentMatchIndex = -1
        if (controller.documentManager) {
            controller.documentManager.set_search_highlights([], -1)
        }
    }

    function performSearch(text) {
        searchText = text
        if (!text || text.length === 0) {
            searchMatches = []
            currentMatchIndex = -1
            if (controller.documentManager) {
                controller.documentManager.set_search_highlights([], -1)
            }
            return
        }

        // Get plain text from document
        const fullDocText = transcriptText.getText(0, transcriptText.length)
        const docText = searchBar.matchCase ? fullDocText : fullDocText.toLowerCase()
        const query = searchBar.matchCase ? text : text.toLowerCase()
        let matches = []
        let pos = 0

        // Word boundary regex pattern for whole words matching
        const isWordChar = (c) => /\w/.test(c)

        while ((pos = docText.indexOf(query, pos)) !== -1) {
            let isMatch = true

            // Check whole words if enabled
            if (searchBar.wholeWords) {
                const beforeChar = pos > 0 ? docText[pos - 1] : ' '
                const afterChar = pos + query.length < docText.length ? docText[pos + query.length] : ' '
                isMatch = !isWordChar(beforeChar) && !isWordChar(afterChar)
            }

            if (isMatch) {
                const rangeIdx = findRangeIndexAt(pos)
                matches.push({
                    charStart: pos,
                    charEnd: pos + query.length,
                    rangeIdx: rangeIdx,
                    timeStart: rangeIdx >= 0 ? wordRanges[rangeIdx].t0 : 0
                })
            }
            pos++
        }

        // Filter OUT matches in cuts unless includeCuts is enabled
        if (!searchBar.includeCuts) {
            const cuts = appController.cuts
            matches = matches.filter(match => {
                if (match.rangeIdx < 0 || match.rangeIdx >= wordRanges.length) return true
                const t0 = wordRanges[match.rangeIdx].t0
                const t1 = wordRanges[match.rangeIdx].t1
                return !isTimeInCuts(t0, t1, cuts)
            })
        }

        searchMatches = matches
        currentMatchIndex = matches.length > 0 ? 0 : -1

        // Apply highlights via document manager (only if highlightAll is enabled)
        updateSearchHighlights()

        // Navigate to first match if found
        if (currentMatchIndex >= 0) {
            navigateToMatch(currentMatchIndex)
        }
    }

    function updateSearchHighlights() {
        if (!controller.documentManager) return
        if (searchBar.highlightAll && searchMatches.length > 0) {
            controller.documentManager.set_search_highlights(searchMatches, currentMatchIndex)
        } else {
            controller.documentManager.set_search_highlights([], -1)
        }
    }

    function navigateToMatch(index) {
        if (index < 0 || index >= searchMatches.length) return

        currentMatchIndex = index
        const match = searchMatches[index]

        // Update highlights to show current match
        updateSearchHighlights()

        // Clear selection anchor so we don't extend from previous selection
        selectionAnchorRangeIdx = -1

        // Select the match text (triggers existing selection logic)
        transcriptText.select(match.charStart, match.charEnd)

        // Scroll into view
        const rect = transcriptText.positionToRectangle(match.charStart)
        const scrollTarget = Math.max(0, rect.y - transcriptScroll.height / 2)
        transcriptScroll.contentItem.contentY = scrollTarget
    }

    function nextMatch() {
        if (searchMatches.length === 0) return
        const newIndex = (currentMatchIndex + 1) % searchMatches.length
        navigateToMatch(newIndex)
    }

    function prevMatch() {
        if (searchMatches.length === 0) return
        const newIndex = currentMatchIndex <= 0 ? searchMatches.length - 1 : currentMatchIndex - 1
        navigateToMatch(newIndex)
    }

    function initTranscript() {
        // Called once when we're ready to show transcript (after VAD, mode is known)
        if (!controller.documentManager) return
        if (!transcriptText || !transcriptText.textDocument) return

        transcriptText.text = ""  // Clear any HTML garbage
        controller.documentManager.set_document(transcriptText.textDocument)
        transcriptReady = true
    }

    // Initialize when we start transcribing or load a project with existing transcript
    onIsTranscribingChanged: {
        if (isTranscribing && !transcriptReady) {
            initTranscript()
        }
    }

    onVisibleChanged: {
        if (!visible) selectionActive = false
    }

    // Reset when controller changes (new transcription session)
    onControllerChanged: {
        transcriptReady = false
        // Initialize document connection when controller is valid
        // This enables rendering even before transcription starts (e.g., loading project)
        if (controller.documentManager) {
            initTranscript()
        }
    }

    onSettingsModelChanged: {
        if (settingsModel) {
            showSpeakerTurns = settingsModel.transcript_show_speaker_turns()
        }
    }

    // Check if a time range overlaps with any cut region
    function isTimeInCuts(t0, t1, cuts) {
        for (let i = 0; i < cuts.length; i++) {
            if (t0 < cuts[i].end && t1 > cuts[i].start) {
                return true
            }
        }
        return false
    }

    // Binary search to find word range containing character position
    function findRangeIndexAt(charPos) {
        const ranges = wordRanges
        if (ranges.length === 0) return -1

        let lo = 0
        let hi = ranges.length - 1

        while (lo <= hi) {
            const mid = Math.floor((lo + hi) / 2)
            const range = ranges[mid]

            if (charPos >= range.start && charPos <= range.end)
                return mid
            else if (charPos < range.start)
                hi = mid - 1
            else
                lo = mid + 1
        }

        // charPos is in a gap between ranges - prefer forward range (what you're clicking into)
        if (lo < ranges.length) return lo
        if (lo > 0) return lo - 1
        return -1
    }

    // Handle selection changes from TextEdit
    property bool adjustingSelection: false

    function handleSelectionChanged() {
        if (adjustingSelection) return
        if (wordRanges.length === 0) return

        const selStart = transcriptText.selectionStart
        const selEnd = transcriptText.selectionEnd

        // Find range indices for selection bounds
        const startIdx = findRangeIndexAt(selStart)
        const endIdx = findRangeIndexAt(selEnd)

        if (startIdx < 0 && endIdx < 0) {
            selectionActive = false
            return
        }

        // Determine anchor vs extent based on tracked anchor word index
        let anchorIdx, extentIdx
        if (selectionAnchorRangeIdx >= 0) {
            // We have a tracked anchor word - use it directly
            anchorIdx = selectionAnchorRangeIdx
            // Extent is whichever end of selection is farther from the anchor
            if (startIdx === anchorIdx) {
                extentIdx = endIdx >= 0 ? endIdx : startIdx
            } else {
                extentIdx = startIdx >= 0 ? startIdx : endIdx
            }
        } else {
            // No anchor tracked, use document order
            anchorIdx = startIdx >= 0 ? startIdx : endIdx
            extentIdx = endIdx >= 0 ? endIdx : startIdx
        }

        const anchorRange = wordRanges[anchorIdx]
        const extentRange = wordRanges[extentIdx]

        // Snap: for forwards selection (anchor <= extent), anchor snaps to start, extent to end
        // For backwards selection (anchor > extent), anchor snaps to end, extent to start
        const forwards = anchorIdx <= extentIdx
        const anchorSnap = forwards ? anchorRange.start : anchorRange.end
        const extentSnap = forwards ? extentRange.end : extentRange.start

        if (anchorSnap !== selStart || extentSnap !== selEnd) {
            adjustingSelection = true
            transcriptText.select(anchorSnap, extentSnap)
            adjustingSelection = false
        }

        // Use min/max for display calculations
        const minIdx = Math.min(anchorIdx, extentIdx)
        const maxIdx = Math.max(anchorIdx, extentIdx)
        const startRange = wordRanges[minIdx]
        const endRange = wordRanges[maxIdx]

        // Store range indices for extended cut calculations
        selectionRangeMinIdx = minIdx
        selectionRangeMaxIdx = maxIdx

        // Store char positions for recalculating rect on scroll
        selectionCharStart = startRange.start
        selectionCharEnd = endRange.end

        // Update selection rect (accounts for scroll offset)
        updateSelectionRect()

        // Mark selection active BEFORE updating time properties, so signal handlers
        // in TranscriptionWorkspaceTab can see selectionActive=true when they fire
        selectionActive = true

        // Update time range - when selectPauses is enabled, extend to fill gaps
        if (selectPauses) {
            selectionTimeStart = minIdx > 0 ? wordRanges[minIdx - 1].t1 : startRange.t0
            selectionTimeEnd = maxIdx < wordRanges.length - 1 ? wordRanges[maxIdx + 1].t0 : endRange.t1
        } else {
            selectionTimeStart = startRange.t0
            selectionTimeEnd = endRange.t1
        }

        // Determine segment ID
        let segId = null
        for (let i = minIdx; i <= maxIdx; ++i) {
            const r = wordRanges[i]
            if (r.isPause) { segId = 0; break }
            if (segId === null) segId = r.segId
            else if (segId !== r.segId) { segId = 0; break }
        }
        selectionSegmentId = segId || 0

        // Notify controller of time range selection
        controller.handle_time_range_selection(selectionTimeStart, selectionTimeEnd)
    }

    Label {
        anchors.centerIn: parent
        text: qsTr("Run transcription to get started.\nSelect text with your mouse to quickly edit your content through text.")
        color: Theme.textSecondary
        horizontalAlignment: Text.AlignHCenter
        visible: !transcriptSurface.hasTranscriptSegments && !transcriptSurface.controllerBusy
    }

    // Placeholder during VAD (preparing phase)
    Label {
        anchors.centerIn: parent
        text: qsTr("Calculating speech segments...")
        color: Theme.textSecondary
        visible: transcriptSurface.controllerBusy && !transcriptSurface.isTranscribing
    }

    ColumnLayout {
        anchors.fill: parent
        spacing: Theme.spacingSm
        visible: transcriptSurface.transcriptReady

        Connections {
            target: transcriptSurface.settingsModel || null
            ignoreUnknownSignals: true
            function onTranscript_show_speaker_turns_changed(enabled) {
                transcriptSurface.showSpeakerTurns = enabled
            }
        }

        Connections {
            target: transcriptSurface.settingsModel || null
            ignoreUnknownSignals: true
            function onTranscript_show_speaker_turns_changed(enabled) {
                transcriptSurface.showSpeakerTurns = enabled
            }
        }

        Row {
            Layout.fillWidth: true
            Layout.fillHeight: true
            spacing: Theme.spacingXs
            layoutDirection: transcriptSurface.isRTL ? Qt.RightToLeft : Qt.LeftToRight

            // Left: timestamp column (moves to right in RTL)
            Item {
                id: timestampColumn
                visible: transcriptSurface.showTimestamps
                width: visible ? 45 : 0
                height: transcriptScroll.height
                clip: true

                Repeater {
                    model: transcriptSurface.timestamps

                    delegate: Label {
                        id: timestampLabel
                        required property var modelData

                        text: transcriptSurface.formatTimestamp(timestampLabel.modelData.time)
                        color: Theme.textSecondary
                        font.pixelSize: transcriptSurface.transcriptFontSize - 2
                        horizontalAlignment: transcriptSurface.isRTL ? Text.AlignLeft : Text.AlignRight
                        width: timestampColumn.width - Theme.spacingXs

                        // Position based on corresponding text position in document
                        y: {
                            void(transcriptText.width)  // Re-evaluate when width changes (affects wrapping)
                            if (!transcriptText.textDocument) return 0
                            const rect = transcriptText.positionToRectangle(timestampLabel.modelData.charPos)
                            const scrollY = transcriptScroll.contentItem ? (transcriptScroll.contentItem as Flickable).contentY : 0
                            return rect.y - scrollY
                        }

                        MouseArea {
                            anchors.fill: parent
                            cursorShape: Qt.PointingHandCursor
                            onClicked: transcriptSurface.appController.seekToTime(timestampLabel.modelData.time)
                        }
                    }
                }
            }

            // Right: transcript ScrollView (moves to left in RTL)
            ScrollView {
                id: transcriptScroll
                width: parent.width - timestampColumn.width - parent.spacing
                height: parent.height
                LayoutMirroring.enabled: transcriptSurface.isRTL

                ScrollBar.horizontal.policy: ScrollBar.AlwaysOff
                ScrollBar.vertical.policy: ScrollBar.AlwaysOn
                ScrollBar.vertical.contentItem: Rectangle {
                    implicitWidth: 8
                    radius: 4
                    color: Qt.rgba(Theme.textSecondary.r, Theme.textSecondary.g, Theme.textSecondary.b, 0.5)
                }

                // Update popover position when scrolling
                Connections {
                    target: transcriptScroll.contentItem
                    function onContentYChanged() {
                        transcriptSurface.updateSelectionRect()
                    }
                }

                TextEdit {
                    id: transcriptText
                    objectName: "transcriptText"
                    width: transcriptScroll.availableWidth
                    leftPadding: Theme.spacingXs
                    rightPadding: Theme.spacingMd
                    readOnly: true
                    wrapMode: TextEdit.Wrap
                    textFormat: TextEdit.RichText
                    selectByMouse: true
                    persistentSelection: true
                    activeFocusOnPress: false  // Don't take keyboard focus - only used for mouse selection
                    color: Theme.textPrimary
                    selectionColor: Theme.textSelection
                    font.pixelSize: transcriptSurface.transcriptFontSize
                    font.family: Qt.platform.os === "windows" ? "Segoe UI Symbol" : font.family
                    horizontalAlignment: transcriptSurface.isRTL ? TextEdit.AlignRight : TextEdit.AlignLeft

                    onSelectedTextChanged: transcriptSurface.handleSelectionChanged()

                    MouseArea {
                        anchors.fill: parent
                        propagateComposedEvents: true
                        cursorShape: Qt.IBeamCursor

                        onPressed: (mouse) => {
                            const charPos = transcriptText.positionAt(mouse.x, mouse.y)
                            transcriptSurface.selectionAnchorRangeIdx = transcriptSurface.findRangeIndexAt(charPos)
                            mouse.accepted = false
                        }
                        onReleased: (mouse) => {
                            transcriptSurface.selectionAnchorRangeIdx = -1
                            mouse.accepted = false
                        }
                    }

                    TapHandler {
                        // Handle double-tap for seek to selection start
                        onDoubleTapped: {
                            if (transcriptSurface.selectionActive) {
                                transcriptSurface.appController.seekToTime(transcriptSurface.selectionTimeStart)
                            }
                        }
                    }
                }
            }
        }

        // Search bar (at bottom to avoid collision with selection popover)
        TranscriptSearchBar {
            id: searchBar
            Layout.fillWidth: true
            Layout.margins: Theme.spacingSm
            visible: transcriptSurface.searchActive
            matchCount: transcriptSurface.searchMatches.length
            currentMatchIndex: transcriptSurface.currentMatchIndex

            onSearchChanged: function(text) { transcriptSurface.performSearch(text) }
            onSearchOptionsChanged: {
                // Re-run search with new matchCase/wholeWords options
                if (transcriptSurface.searchText) {
                    transcriptSurface.performSearch(transcriptSurface.searchText)
                }
            }
            onHighlightOptionsChanged: {
                // Just update highlights visually, don't change selection
                transcriptSurface.updateSearchHighlights()
            }
            onNextMatch: transcriptSurface.nextMatch()
            onPrevMatch: transcriptSurface.prevMatch()
            onClosed: transcriptSurface.closeSearch()
        }
    }

    // Connect to document manager for incremental updates
    Connections {
        target: transcriptSurface.controller.documentManager

        function onSegment_rendered(segmentId, charStart, ranges) {
            // Extend wordRanges array with new ranges (concat is faster than slice+push loop)
            transcriptSurface.wordRanges = transcriptSurface.wordRanges.concat(ranges)

            // Update segCharStart
            var newSegStarts = Object.assign({}, transcriptSurface.segCharStart)
            newSegStarts[segmentId] = charStart
            transcriptSurface.segCharStart = newSegStarts
        }

        function onTimestamp_added(time, charPos) {
            // Add timestamp marker for left column
            transcriptSurface.timestamps = transcriptSurface.timestamps.concat([{time: time, charPos: charPos}])
        }

        function onDocument_cleared() {
            transcriptSurface.clearState()
        }

        function onContent_cleared() {
            transcriptSurface.clearRanges()
        }
    }

    TranscriptSelectionPopover {
        id: selectionPopover
        controller: transcriptSurface.controller
        appController: transcriptSurface.appController
        selectionActive: transcriptSurface.selectionActive
        selectionInView: transcriptSurface.selectionInView
        selectionRect: transcriptSurface.selectionRect
        selectionTimeStart: transcriptSurface.selectionTimeStart
        selectionTimeEnd: transcriptSurface.selectionTimeEnd
        selectionSegmentId: transcriptSurface.selectionSegmentId
        sourceItem: transcriptSurface
        onDismissed: transcriptSurface.selectionActive = false
    }

    onSelectionActiveChanged: {
        if (selectionActive && showPopup) selectionPopover.open()
        else selectionPopover.close()
    }

    onShowPopupChanged: {
        if (!showPopup) selectionPopover.close()
        else if (selectionActive) selectionPopover.open()
    }

    // Update font size when settings change
    Connections {
        target: transcriptSurface.settingsModel
        function onTranscript_font_size_changed(size) {
            transcriptSurface.transcriptFontSize = size
        }
        function onTranscript_show_timestamps_changed(enabled) {
            transcriptSurface.showTimestamps = enabled
        }
        function onTranscript_show_popup_changed(enabled) {
            transcriptSurface.showPopup = enabled
        }
        function onTranscript_select_pauses_changed(enabled) {
            transcriptSurface.selectPauses = enabled
        }
    }

    // Re-run search when cuts change (if filtering out cuts)
    Connections {
        target: transcriptSurface.appController
        function onCutsChanged() {
            if (transcriptSurface.searchActive && !searchBar.includeCuts) {
                transcriptSurface.performSearch(transcriptSurface.searchText)
            }
        }
    }

}
