pragma ComponentBehavior: Bound
import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import SMC.Controllers 1.0
import styles 1.0
import components.controls 1.0

Frame {
    id: root
    required property QmlTranscriptionController controller

    Layout.fillWidth: true
    padding: Theme.spacingLg

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surfaceStrong
        border.color: Theme.surfaceBorder
    }

    contentItem: ColumnLayout {
        spacing: Theme.spacingMd

        RowLayout {
            spacing: Theme.spacingLg

            ComboBox {
                id: modelSelector
                Layout.preferredWidth: 220
                textRole: "label"
                valueRole: "key"
                model: root.controller && root.controller.modelOptions ? root.controller.modelOptions : []
                property bool internalChange: false

                function syncFromController() {
                    const options = root.controller && root.controller.modelOptions ? root.controller.modelOptions : []
                    const key = root.controller && root.controller.modelSize !== undefined ? root.controller.modelSize : ""
                    for (let i = 0; i < options.length; ++i) {
                        if (options[i].key === key) {
                            internalChange = true
                            currentIndex = i
                            internalChange = false
                            return
                        }
                    }
                    if (options.length > 0) {
                        internalChange = true
                        currentIndex = 0
                        internalChange = false
                    }
                }

                delegate: ItemDelegate {
                    required property var model
                    width: parent.width
                    text: model.label + (model.size ? " (" + model.size + ")" : "")
                }

                onActivated: {
                    if (internalChange)
                        return
                    if (!root.controller)
                        return
                    if (currentIndex >= 0 && currentIndex < model.length)
                        root.controller.set_model_size(model[currentIndex].key)
                }

                Component.onCompleted: syncFromController()

                Connections {
                    target: root.controller || null
                    ignoreUnknownSignals: true
                    function onModelSizeChanged() {
                        modelSelector.syncFromController()
                    }
                }
            }

            Switch {
                id: gpuSwitch
                text: qsTr("Use GPU")
                checked: root.controller && root.controller.gpuMode
                onToggled: {
                    if (root.controller)
                        root.controller.set_gpu_mode(checked)
                }

                Connections {
                    target: root.controller || null
                    ignoreUnknownSignals: true
                    function onGpuModeChanged(value) {
                        if (gpuSwitch.checked !== value) {
                            gpuSwitch.checked = value
                        }
                    }
                }
            }

            Item { Layout.fillWidth: true }
        }

        ColumnLayout {
            spacing: Theme.spacingXs
            Layout.fillWidth: true

            Label {
                text: qsTr("Tracks to transcribe")
                color: Theme.textSecondary
            }

            ColumnLayout {
                Layout.fillWidth: true
                spacing: Theme.spacingXs

                Repeater {
                    model: root.controller && root.controller.tracks ? root.controller.tracks : []
                    delegate: CheckDelegate {
                        id: checkDelegate
                        required property var modelData
                        Layout.fillWidth: true
                        text: modelData.label
                        checked: modelData.selected
                        indicator: Rectangle {
                            implicitWidth: 20
                            implicitHeight: 20
                            radius: 4
                            border.width: 0
                            color: checkDelegate.modelData.color
                        }
                        background: Rectangle {
                            color: Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, checkDelegate.checked ? 0.08 : 0.0)
                            radius: Theme.panelRadius - 4
                        }
                        onToggled: {
                            if (root.controller)
                                root.controller.set_track_selected(modelData.index, checked)
                        }
                    }
                }

                Label {
                    visible: !root.controller || !root.controller.tracks || root.controller.tracks.length === 0
                    text: qsTr("Load media to select tracks for transcription.")
                    color: Theme.textSecondary
                }
            }
        }

        RowLayout {
            Layout.fillWidth: true
            spacing: Theme.spacingSm

            Item { Layout.fillWidth: true }

            Button {
                text: qsTr("Cancel")
                enabled: root.controller && root.controller.busy
                onClicked: {
                    if (root.controller)
                        root.controller.cancel_transcription()
                }
            }

            PrimaryButton {
                text: root.controller && root.controller.busy ? qsTr("Working…") : qsTr("Start Transcription")
                enabled: root.controller ? !root.controller.busy : false
                onClicked: {
                    if (root.controller)
                        root.controller.start_transcription()
                }
            }
        }
    }
}
