pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Dialogs
import Qt5Compat.GraphicalEffects
import SMC.Controllers 1.0
import styles 1.0
import components.controls 1.0
import components.forms 1.0 as Forms
import components.notifications 1.0
import components.layout 1.0
import "../../../utils" as Utils

Frame {
    id: root
    required property QmlTranscriptionController controller
    required property var licenseManager
    required property var languageController
    required property var toastManager
    property var settingsModel: null

    // View state for switching between main sidebar and translation panel
    property bool showTranslationView: false

    // Reset translation view when transcript is reset (e.g., media changes)
    Connections {
        target: root.controller || null
        ignoreUnknownSignals: true
        function onTranscriptSegmentsChanged() {
            // If transcript was cleared, close translation panel
            if (root.controller && root.controller.transcriptSegments.length === 0) {
                root.showTranslationView = false
            }
        }
    }

    // Transcript exists when we have segments and aren't actively processing
    readonly property bool hasTranscript: root.controller &&
        !root.controller.busy &&
        !root.controller.downloadingModel &&
        root.controller.transcriptSegments.length > 0

    signal searchRequested()
    signal feedbackRequested()

    // Sync track selection to controller before transcription starts
    function syncTrackSelection() {
        if (!root.controller || !root.controller.tracks) return
        const selected = trackSelectionPanel.getSelectedIndices()
        const tracks = root.controller.tracks
        for (let i = 0; i < tracks.length; i++) {
            const isSelected = selected.indexOf(tracks[i].index) >= 0
            root.controller.set_track_selected(tracks[i].index, isSelected)
        }
    }

    // Visual properties
    padding: 0 // We'll handle padding internally for better control

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surface
        border.color: Theme.surfaceBorder
    }

    contentItem: StackLayout {
        currentIndex: root.showTranslationView ? 1 : 0

        // Index 0: Main transcription sidebar
        ColumnLayout {
            spacing: 0

            SidebarHeader {
                title: qsTr("Transcription")
            }

            ScrollView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                // Settings Section
                ColumnLayout {
                    id: settingsSection
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd

                    Label {
                        text: qsTr("Settings")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    // Model selection - simplified for end users
                    // Maps user-friendly choices to actual whisper model keys
                    QtObject {
                        id: modelMapping

                        // English models are one size smaller (medium.en is largest)
                        readonly property var englishModels: ({
                            "small": "base.en",
                            "medium": "small.en",
                            "large": "medium.en"
                        })

                        readonly property var multilingualModels: ({
                            "small": "small",
                            "medium": "medium",
                            "large": "large-v3"
                        })

                        function getModelKey(isEnglish, size) {
                            return isEnglish ? englishModels[size] : multilingualModels[size]
                        }

                        // Reverse lookup: from model key to user selections
                        function fromModelKey(key) {
                            // Check English models
                            for (const size in englishModels) {
                                if (englishModels[size] === key) {
                                    return { isEnglish: true, size: size }
                                }
                            }
                            // Check multilingual models
                            for (const size in multilingualModels) {
                                if (multilingualModels[size] === key) {
                                    return { isEnglish: false, size: size }
                                }
                            }
                            // Default
                            return { isEnglish: true, size: "medium" }
                        }
                    }

                    // Transcription engine selection (3-way: English, Multilingual, Cloud)
                    Label {
                        text: qsTr("Transcription Engine")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        Button {
                            id: englishBtn
                            objectName: "englishEngineButton"
                            Layout.fillWidth: true
                            text: qsTr("English")
                            checkable: true
                            checked: root.controller && root.controller.transcriptionBackend === "local" && settingsSection.isEnglishModel
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: englishBtn.checked
                                    ? Theme.accent
                                    : (englishBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: englishBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: englishBtn.text
                                color: englishBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                if (root.controller) {
                                    root.controller.set_transcription_backend("local")
                                    settingsSection.updateModel()
                                    if (root.controller) {
                                        root.controller.record_milestone("transcription_engine")
                                    }
                                }
                            }
                        }

                        Button {
                            id: multilingualBtn
                            objectName: "multilingualEngineButton"
                            Layout.fillWidth: true
                            text: qsTr("Multilingual")
                            checkable: true
                            checked: root.controller && root.controller.transcriptionBackend === "local" && !settingsSection.isEnglishModel
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: multilingualBtn.checked
                                    ? Theme.accent
                                    : (multilingualBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: multilingualBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: multilingualBtn.text
                                color: multilingualBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                if (root.controller) {
                                    root.controller.set_transcription_backend("local")
                                    settingsSection.updateModel()
                                    if (root.controller) {
                                        root.controller.record_milestone("transcription_engine")
                                    }
                                }
                            }
                        }

                        Button {
                            id: cloudBtn
                            objectName: "cloudEngineButton"
                            Layout.fillWidth: true
                            text: qsTr("Cloud")
                            checkable: true
                            checked: root.controller && root.controller.transcriptionBackend === "cloud"
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: cloudBtn.checked
                                    ? Theme.accent
                                    : (cloudBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: cloudBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: RowLayout {
                                spacing: 4
                                Item { Layout.fillWidth: true }
                                Item {
                                    Layout.preferredWidth: 14
                                    Layout.preferredHeight: 14
                                    visible: !root.licenseManager.isValid()

                                    Image {
                                        id: crownImage
                                        anchors.fill: parent
                                        source: Theme.iconUrl("crown")
                                        sourceSize.width: 14
                                        sourceSize.height: 14
                                        visible: false
                                    }
                                    ColorOverlay {
                                        anchors.fill: parent
                                        source: crownImage
                                        color: cloudBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                    }
                                }
                                Label {
                                    text: cloudBtn.text
                                    color: cloudBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                }
                                Item { Layout.fillWidth: true }
                            }

                            onClicked: {
                                if (root.controller) {
                                    if (!root.licenseManager.isValid()) {
                                        // Restore previous selection - autoExclusive already checked this button
                                        cloudBtn.checked = false
                                        if (settingsSection.isEnglishModel) {
                                            englishBtn.checked = true
                                        } else {
                                            multilingualBtn.checked = true
                                        }
                                        premiumFeaturePopup.open()
                                    } else {
                                        root.controller.set_transcription_backend("cloud")
                                    }
                                }
                            }
                        }
                    }

                    // Track whether English model is selected (for button state)
                    property bool isEnglishModel: true

                    Connections {
                        target: root.controller || null
                        ignoreUnknownSignals: true
                        function onTranscriptionBackendChanged(backend) {
                            // Update button states when backend changes
                            cloudBtn.checked = (backend === "cloud")
                            if (backend === "local") {
                                settingsSection.syncFromController()
                            }
                        }
                    }

                    // Cloud provider selection (only when cloud is selected)
                    Label {
                        text: qsTr("Cloud Provider")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                        Layout.topMargin: Theme.spacingSm
                        visible: cloudBtn.checked
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm
                        visible: cloudBtn.checked

                        Button {
                            id: assemblyaiBtn
                            Layout.fillWidth: true
                            checkable: true
                            checked: root.controller && root.controller.cloudProvider === "assemblyai"
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: assemblyaiBtn.checked
                                    ? Theme.accent
                                    : (assemblyaiBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: assemblyaiBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: "AssemblyAI"
                                color: assemblyaiBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                if (root.controller) {
                                    root.controller.set_cloud_provider("assemblyai")
                                }
                            }
                        }

                        Button {
                            id: elevenlabsBtn
                            Layout.fillWidth: true
                            checkable: true
                            checked: root.controller && root.controller.cloudProvider === "elevenlabs"
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: elevenlabsBtn.checked
                                    ? Theme.accent
                                    : (elevenlabsBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: elevenlabsBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: "ElevenLabs"
                                color: elevenlabsBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                if (root.controller) {
                                    root.controller.set_cloud_provider("elevenlabs")
                                }
                            }
                        }
                    }

                    // Disfluencies option (AssemblyAI only, English or Auto only)
                    RowLayout {
                        Layout.fillWidth: true
                        Layout.topMargin: Theme.spacingSm
                        spacing: Theme.spacingSm
                        visible: cloudBtn.checked && assemblyaiBtn.checked &&
                                 (languagePicker.code === "" || languagePicker.code === "en")

                        Switch {
                            id: disfluenciesSwitch
                            checked: root.controller ? root.controller.disfluencies : false
                            onToggled: {
                                if (root.controller) {
                                    root.controller.set_disfluencies(checked)
                                }
                            }

                            Connections {
                                target: root.controller || null
                                ignoreUnknownSignals: true
                                function onDisfluenciesChanged(value) {
                                    if (disfluenciesSwitch.checked !== value) {
                                        disfluenciesSwitch.checked = value
                                    }
                                }
                            }
                        }

                        Label {
                            Layout.fillWidth: true
                            text: qsTr("Detect filler words (only works with English content)")
                            color: Theme.textSecondary
                            font.pixelSize: 13
                            wrapMode: Text.WordWrap
                        }
                    }

                    // Transcription language dropdown (multilingual local or cloud)
                    Label {
                        text: qsTr("Transcription language")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                        Layout.topMargin: Theme.spacingSm
                        visible: multilingualBtn.checked || cloudBtn.checked
                    }

                    Forms.LanguagePickerField {
                        id: languagePicker
                        objectName: "languageDropdown"
                        Layout.fillWidth: true
                        visible: multilingualBtn.checked || cloudBtn.checked
                        settingsModel: root.settingsModel
                        includeAutonyms: true
                        accessibleName: qsTr("Transcription language")

                        function syncFromController() {
                            languagePicker.code = root.controller ? root.controller.language : ""
                        }

                        Component.onCompleted: {
                            languagePicker.syncFromController()
                        }

                        onCodeSelected: function(langCode) {
                            if (root.controller) {
                                root.controller.set_language(langCode)
                            }
                            if (root.controller) {
                                root.controller.record_milestone("transcription_language")
                            }
                            // Note: We don't write back to settings here.
                            // Settings holds the default, sidebar is per-session.
                        }
                    }

                    Connections {
                        target: root.controller || null
                        ignoreUnknownSignals: true
                        function onLanguageChanged() {
                            languagePicker.syncFromController()
                        }
                    }

                    // Model size selection (only for local backend)
                    Label {
                        text: qsTr("Model size")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                        Layout.topMargin: Theme.spacingSm
                        visible: !cloudBtn.checked
                    }

                    Label {
                        text: qsTr("Larger models are slower but usually better quality")
                        font.pixelSize: 11
                        color: Theme.textSecondary
                        opacity: 0.7
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                        visible: !cloudBtn.checked
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm
                        visible: !cloudBtn.checked

                        Button {
                            id: smallBtn
                            objectName: "smallModelButton"
                            Layout.fillWidth: true
                            checkable: true
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: smallBtn.checked
                                    ? Theme.accent
                                    : (smallBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: smallBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: qsTr("Small")
                                color: smallBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                settingsSection.updateModel()
                                if (root.controller) {
                                    root.controller.record_milestone("model_size")
                                }
                            }
                        }

                        Button {
                            id: mediumBtn
                            objectName: "mediumModelButton"
                            Layout.fillWidth: true
                            checkable: true
                            checked: true
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: mediumBtn.checked
                                    ? Theme.accent
                                    : (mediumBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: mediumBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: qsTr("Medium")
                                color: mediumBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                settingsSection.updateModel()
                                if (root.controller) {
                                    root.controller.record_milestone("model_size")
                                }
                            }
                        }

                        Button {
                            id: largeBtn
                            objectName: "largeModelButton"
                            Layout.fillWidth: true
                            checkable: true
                            autoExclusive: true

                            background: Rectangle {
                                radius: Theme.panelRadius - 4
                                color: largeBtn.checked
                                    ? Theme.accent
                                    : (largeBtn.hovered ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.08) : Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.04))
                                border.width: largeBtn.checked ? 0 : 1
                                border.color: Theme.surfaceBorder
                            }

                            contentItem: Label {
                                text: qsTr("Large")
                                color: largeBtn.checked ? Theme.textOnAccent : Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }

                            onClicked: {
                                settingsSection.updateModel()
                                if (root.controller) {
                                    root.controller.record_milestone("model_size")
                                }
                            }
                        }
                    }

                    function updateModel() {
                        if (!root.controller) return
                        const isEnglish = englishBtn.checked
                        settingsSection.isEnglishModel = isEnglish
                        const size = smallBtn.checked ? "small" : (mediumBtn.checked ? "medium" : "large")
                        const modelKey = modelMapping.getModelKey(isEnglish, size)
                        root.controller.set_model_size(modelKey)
                    }

                    function syncFromController() {
                        if (!root.controller) return
                        const key = root.controller.modelSize || ""
                        const mapping = modelMapping.fromModelKey(key)

                        // Update isEnglishModel property
                        settingsSection.isEnglishModel = mapping.isEnglish

                        // Update button states only if local backend
                        if (root.controller.transcriptionBackend === "local") {
                            englishBtn.checked = mapping.isEnglish
                            multilingualBtn.checked = !mapping.isEnglish
                            cloudBtn.checked = false
                        }

                        smallBtn.checked = (mapping.size === "small")
                        mediumBtn.checked = (mapping.size === "medium")
                        largeBtn.checked = (mapping.size === "large")
                    }

                    function initializeLanguageFromSettings() {
                        // Load saved language from settings and apply to controller
                        if (root.settingsModel && root.controller) {
                            var savedLang = root.settingsModel.default_transcription_language()
                            root.controller.set_language(savedLang)
                        }
                    }

                    Component.onCompleted: {
                        syncFromController()
                        initializeLanguageFromSettings()
                    }

                    Connections {
                        target: root.controller || null
                        ignoreUnknownSignals: true
                        function onModelSizeChanged() {
                            settingsSection.syncFromController()
                        }
                    }

                    // Download section - shown when model needs downloading or is downloading (local only)
                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm
                        visible: !cloudBtn.checked && root.controller && (root.controller.needsDownload || root.controller.downloadingModel)

                        Rectangle {
                            Layout.fillWidth: true
                            Layout.preferredHeight: downloadContent.implicitHeight + Theme.spacingMd * 2
                            radius: Theme.panelRadius - 4
                            color: Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.1)
                            border.color: Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.3)
                            border.width: 1

                            ColumnLayout {
                                id: downloadContent
                                anchors.fill: parent
                                anchors.margins: Theme.spacingMd
                                spacing: Theme.spacingSm

                                Label {
                                    text: root.controller && root.controller.downloadMessage ? root.controller.downloadMessage : qsTr("Model needs to be downloaded")
                                    color: Theme.textSecondary
                                    font.pixelSize: 12
                                    wrapMode: Text.WordWrap
                                    Layout.fillWidth: true
                                }

                                ProgressBar {
                                    Layout.fillWidth: true
                                    from: 0
                                    to: 100
                                    value: root.controller ? root.controller.progress : 0
                                    visible: root.controller && root.controller.downloadingModel
                                }

                                PrimaryButton {
                                    objectName: "downloadModelButton"
                                    text: root.controller && root.controller.downloadingModel ? qsTr("Downloading…") : qsTr("Download Model")
                                    enabled: root.controller && !root.controller.downloadingModel
                                    Layout.fillWidth: true
                                    visible: !root.controller || !root.controller.downloadingModel
                                    onClicked: {
                                        if (root.controller)
                                            root.controller.download_model()
                                    }
                                }
                            }
                        }
                    }
                }

                // Tracks Section - only show if more than 1 track
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 1
                    color: Theme.surfaceBorder
                    visible: trackSelectionPanel.shouldShow && !root.controller.busy && !root.hasTranscript
                }

                ColumnLayout {
                    id: tracksSection
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd
                    visible: trackSelectionPanel.shouldShow && !root.controller.busy && !root.hasTranscript

                    Label {
                        text: qsTr("Choose audio tracks to transcribe")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    Forms.TrackSelectionPanel {
                        id: trackSelectionPanel
                        Layout.fillWidth: true
                        tracks: root.controller ? root.controller.tracks : []
                        settingsModel: root.settingsModel
                        helperText: qsTr("Select which audio tracks to transcribe")
                        onSelectionChanged: {
                            // Sync track selection to controller so VAD cache uses correct key
                            root.syncTrackSelection()
                        }
                    }
                }

                // Cost estimation section (cloud only, before transcription starts)
                ColumnLayout {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    Layout.topMargin: 0
                    spacing: Theme.spacingSm
                    visible: cloudBtn.checked && !root.controller.busy && !root.hasTranscript

                    // Show estimate if VAD segments exist
                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: 4
                        visible: root.controller && root.controller.hasVadSegments && !root.controller.isCalculatingEstimate

                        Label {
                            property int credits: Math.round(root.controller ? root.controller.estimatedCredits : 0)
                            property string dollars: (credits / 4000).toFixed(2)
                            text: qsTr("Estimated cost: %1 credits ($%2)").arg(credits).arg(dollars)
                            color: Theme.textPrimary
                            font.pixelSize: 13
                            font.weight: Font.DemiBold
                        }

                        Label {
                            property int totalSeconds: Math.round(root.controller ? root.controller.speechDuration : 0)
                            property int minutes: Math.floor(totalSeconds / 60)
                            property int seconds: totalSeconds % 60
                            text: qsTr("Based on %1m %2s of detected speech").arg(minutes).arg(seconds)
                            color: Theme.textSecondary
                            font.pixelSize: 12
                            wrapMode: Text.WordWrap
                            Layout.fillWidth: true
                        }

                        // Percentage of remaining credits
                        Label {
                            property int remaining: root.licenseManager ?
                                (root.licenseManager.creditsLimit - root.licenseManager.creditsUsed) : 0
                            property int estimated: Math.round(root.controller ? root.controller.estimatedCredits : 0)
                            property real percentage: remaining > 0 ? (estimated / remaining * 100) : 0
                            visible: root.licenseManager &&
                                     !root.licenseManager.creditsLoading &&
                                     remaining > 0
                            text: qsTr("This is %1% of your remaining credits").arg(percentage.toFixed(1))
                            color: Theme.textSecondary
                            font.pixelSize: 12
                            wrapMode: Text.WordWrap
                            Layout.fillWidth: true
                        }

                        // Warning: insufficient credits
                        Rectangle {
                            property int remaining: root.licenseManager ?
                                (root.licenseManager.creditsLimit - root.licenseManager.creditsUsed) : 0
                            property int estimated: Math.round(root.controller ? root.controller.estimatedCredits : 0)
                            visible: root.licenseManager &&
                                     !root.licenseManager.creditsLoading &&
                                     estimated > remaining
                            Layout.fillWidth: true
                            Layout.topMargin: Theme.spacingSm
                            radius: Theme.panelRadius - 4
                            color: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, 0.15)
                            border.color: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, 0.4)
                            border.width: 1
                            implicitHeight: warningLabel.implicitHeight + Theme.spacingSm * 2

                            Label {
                                id: warningLabel
                                anchors.fill: parent
                                anchors.margins: Theme.spacingSm
                                text: qsTr("Insufficient credits for this transcription.")
                                color: Theme.statusDanger
                                font.pixelSize: 12
                                font.weight: Font.DemiBold
                                wrapMode: Text.WordWrap
                            }
                        }
                    }

                    // Show calculate button (visible when not calculating)
                    Button {
                        Layout.fillWidth: true
                        text: qsTr("Calculate credits cost estimate")
                        visible: root.controller && !root.controller.isCalculatingEstimate
                        font.pixelSize: 12
                        onClicked: {
                            if (root.controller) {
                                root.controller.calculate_cost_estimate()
                            }
                            if (root.licenseManager) {
                                root.licenseManager.refreshCredits()
                            }
                        }
                    }

                    // Show spinner while calculating
                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm
                        visible: root.controller && root.controller.isCalculatingEstimate

                        BusyIndicator {
                            Layout.preferredWidth: 16
                            Layout.preferredHeight: 16
                            running: visible
                        }

                        Label {
                            text: qsTr("Analyzing audio...")
                            color: Theme.textSecondary
                            font.pixelSize: 12
                        }
                    }
                }

                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 1
                    color: Theme.surfaceBorder
                    visible: progressSection.visible
                }

                // Progress Section - only show during actual transcription (not model download)
                ColumnLayout {
                    id: progressSection
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd
                    visible: root.controller && !root.controller.downloadingModel && (root.controller.busy || (root.controller.progress > 0 && root.controller.progress < 100))

                    Label {
                        text: qsTr("Progress")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    Label {
                        text: root.controller && root.controller.status ? root.controller.status : qsTr("Idle")
                        color: Theme.textSecondary
                        font.pixelSize: 13
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }

                    ProgressBar {
                        Layout.fillWidth: true
                        from: 0
                        to: 100
                        value: root.controller && root.controller.progress !== undefined ? root.controller.progress : 0
                    }
                }

                // Transcript Tools Section - visible after transcription is complete
                Rectangle {
                    Layout.fillWidth: true
                    Layout.preferredHeight: 1
                    color: Theme.surfaceBorder
                    visible: transcriptToolsSection.visible
                }

                ColumnLayout {
                    id: transcriptToolsSection
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    spacing: Theme.spacingMd
                    visible: root.hasTranscript

                    Label {
                        text: qsTr("Transcript Tools")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    // Cut pauses section
                    Label {
                        text: qsTr("Cut Pauses by length:")
                        font.pixelSize: 12
                        color: Theme.textSecondary
                        Layout.topMargin: Theme.spacingSm
                    }

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        SecondaryButton {
                            Layout.fillWidth: true
                            text: qsTr("All")
                            onClicked: root.controller.cut_pauses("all")
                        }

                        SecondaryButton {
                            Layout.fillWidth: true
                            text: qsTr("Long")
                            onClicked: root.controller.cut_pauses("long")
                        }

                        SecondaryButton {
                            Layout.fillWidth: true
                            text: qsTr("Short")
                            onClicked: root.controller.cut_pauses("short")
                        }
                    }

                    Rectangle {
                        Layout.fillWidth: true
                        Layout.preferredHeight: 1
                        color: Theme.surfaceBorder
                        opacity: 0.5
                    }

                    // Search transcript button
                    SecondaryButton {
                        Layout.fillWidth: true
                        text: qsTr("Search in Transcript")
                        onClicked: root.searchRequested()
                    }

                    // Translate transcript button
                    SecondaryButton {
                        id: translateBtn
                        Layout.fillWidth: true
                        text: qsTr("Translate Transcript")

                        onClicked: {
                            if (!root.licenseManager.isValid()) {
                                translationPremiumPopup.open()
                            } else {
                                root.showTranslationView = true
                            }
                        }

                        contentItem: RowLayout {
                            spacing: 4
                            Item { Layout.fillWidth: true }
                            Item {
                                Layout.preferredWidth: 14
                                Layout.preferredHeight: 14
                                visible: !root.licenseManager.isValid()

                                Image {
                                    id: translateCrownImage
                                    anchors.fill: parent
                                    source: Theme.iconUrl("crown")
                                    sourceSize.width: 14
                                    sourceSize.height: 14
                                    visible: false
                                }
                                ColorOverlay {
                                    anchors.fill: parent
                                    source: translateCrownImage
                                    color: Theme.textPrimary
                                }
                            }
                            Label {
                                text: translateBtn.text
                                color: Theme.textPrimary
                                horizontalAlignment: Text.AlignHCenter
                                verticalAlignment: Text.AlignVCenter
                            }
                            Item { Layout.fillWidth: true }
                        }
                    }

                    // Export subtitles button
                    SecondaryButton {
                        Layout.fillWidth: true
                        text: qsTr("Export Transcript to .srt File")
                        onClicked: {
                            if (root.controller)
                                root.controller.export_subtitles()
                        }
                    }

                    // Embed on export option
                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingSm

                        Switch {
                            id: embedSubtitlesSwitch
                            checked: root.controller ? root.controller.embedSubtitles : false
                            onToggled: {
                                if (root.controller) {
                                    root.controller.set_embed_subtitles(checked)
                                }
                            }

                            Connections {
                                target: root.controller || null
                                ignoreUnknownSignals: true
                                function onEmbedSubtitlesChanged(value) {
                                    if (embedSubtitlesSwitch.checked !== value) {
                                        embedSubtitlesSwitch.checked = value
                                    }
                                }
                            }
                        }

                        Label {
                            Layout.fillWidth: true
                            text: qsTr("Add transcript and translations as a subtitle tracks on video export")
                            color: Theme.textSecondary
                            font.pixelSize: 13
                            wrapMode: Text.WordWrap
                        }
                    }
                }
            }
        }

        // Footer Actions
        ColumnLayout {
            Layout.fillWidth: true
            Layout.margins: Theme.spacingLg
            Layout.topMargin: 0
            spacing: Theme.spacingSm

            // Feedback link - always visible so users can report issues
            Label {
                text: qsTr("Report transcription issue")
                color: Theme.accent
                font.pixelSize: 13
                font.underline: feedbackMouseArea.containsMouse

                MouseArea {
                    id: feedbackMouseArea
                    anchors.fill: parent
                    cursorShape: Qt.PointingHandCursor
                    hoverEnabled: true
                    onClicked: root.feedbackRequested()
                }
            }

            Rectangle {
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingSm
                Layout.preferredHeight: 1
                color: Theme.surfaceBorder
            }

            // No speech detected info box
            Rectangle {
                Layout.fillWidth: true
                Layout.bottomMargin: Theme.spacingSm
                visible: root.controller && root.controller.noSpeechDetected
                radius: Theme.panelRadius - 4
                color: Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.1)
                border.color: Qt.rgba(Theme.accent.r, Theme.accent.g, Theme.accent.b, 0.3)
                border.width: 1
                implicitHeight: noSpeechContent.implicitHeight + Theme.spacingMd * 2

                ColumnLayout {
                    id: noSpeechContent
                    anchors.fill: parent
                    anchors.margins: Theme.spacingMd
                    spacing: Theme.spacingSm

                    Label {
                        text: qsTr("No speech detected")
                        font.weight: Font.DemiBold
                        color: Theme.textPrimary
                    }

                    Label {
                        text: qsTr("No speech was found in the selected audio.")
                        color: Theme.textSecondary
                        font.pixelSize: 12
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }

                    Label {
                        visible: root.controller && root.controller.tracks && root.controller.tracks.length > 1
                        text: qsTr("Try selecting a different audio track.")
                        color: Theme.textSecondary
                        font.pixelSize: 12
                        wrapMode: Text.WordWrap
                        Layout.fillWidth: true
                    }
                }
            }

            PrimaryButton {
                objectName: "startTranscriptionButton"
                Layout.fillWidth: true
                text: root.controller && root.controller.busy ? qsTr("Transcribing…") : qsTr("Start Transcription")
                enabled: root.controller ? !root.controller.busy : false
                onClicked: {
                    if (root.controller) {
                        root.syncTrackSelection()
                        root.controller.start_transcription()
                    }
                }
            }

            Button {
                Layout.fillWidth: true
                text: qsTr("Stop")
                visible: root.controller && root.controller.busy
                onClicked: {
                    if (root.controller)
                        root.controller.stop_transcription()
                }
            }

            SecondaryButton {
                Layout.fillWidth: true
                text: qsTr("Reset")
                visible: root.controller && !root.controller.busy && root.hasTranscript
                onClicked: {
                    if (root.controller)
                        root.controller.reset_transcript()
                }
            }
        }
        } // End of index 0: Main transcription sidebar ColumnLayout

        // Index 1: Translation panel
        TranslationPanel {
            controller: root.controller
            licenseManager: root.licenseManager
            languageController: root.languageController
            toastManager: root.toastManager
            settingsModel: root.settingsModel
            onCloseRequested: root.showTranslationView = false
        }
    } // End of StackLayout

    // Subtitle export dialog
    FileDialog {
        id: subtitleExportDialog
        title: qsTr("Export Subtitles")
        fileMode: FileDialog.SaveFile
        nameFilters: [
            qsTr("SRT Subtitle") + " (*.srt)",
            qsTr("WebVTT Subtitle") + " (*.vtt)",
            qsTr("All Files") + " (*)"
        ]

        function openWithPath(path) {
            if (path) {
                currentFolder = Utils.PathUtils.pathToFolderUrl(path);
                selectedFile = Utils.PathUtils.pathToUrl(path);
            }
            open();
        }

        onAccepted: {
            let path = Utils.PathUtils.urlToPath(selectedFile);
            if (root.controller && path) {
                // Add extension if missing based on selected filter
                const lowerPath = path.toLowerCase();
                const filterName = String(selectedNameFilter.name);
                if (filterName.indexOf(".srt") >= 0 && !lowerPath.endsWith(".srt")) {
                    path += ".srt";
                } else if (filterName.indexOf(".vtt") >= 0 && !lowerPath.endsWith(".vtt")) {
                    path += ".vtt";
                }
                root.controller.export_subtitles_to_path(path);
            }
        }
    }

    Connections {
        target: root.controller || null
        ignoreUnknownSignals: true

        function onExportSubtitlesRequested(suggestedPath) {
            subtitleExportDialog.openWithPath(suggestedPath);
        }
    }

    // Premium feature popup for cloud transcription
    PremiumFeaturePopup {
        id: premiumFeaturePopup
        parent: Overlay.overlay
        anchors.centerIn: parent
        featureName: qsTr("Cloud Transcription")
        featureDescription: qsTr("Fast, accurate transcription powered by cloud AI.")
        featureBenefits: [
            qsTr("Up to 10x faster than local transcription"),
            qsTr("Works on any machine, no GPU needed"),
            qsTr("High accuracy for any audio quality")
        ]
        licenseManager: root.licenseManager
        languageController: root.languageController

        onActivationSucceeded: {
            root.toastManager.success(qsTr("License activated successfully!"), "")
            if (root.controller) {
                root.controller.set_transcription_backend("cloud")
            }
            close()
        }
    }

    // Premium feature popup for translation (when user clicks translate without license)
    PremiumFeaturePopup {
        id: translationPremiumPopup
        parent: Overlay.overlay
        anchors.centerIn: parent
        featureName: qsTr("Translation")
        featureDescription: qsTr("Translate your transcripts to 40+ languages using cloud AI.")
        featureBenefits: [
            qsTr("Fast cloud-powered translation"),
            qsTr("Subtitles stay synced with video"),
            qsTr("Professional quality results")
        ]
        licenseManager: root.licenseManager
        languageController: root.languageController

        onActivationSucceeded: {
            root.toastManager.success(qsTr("License activated successfully!"), "")
            close()
            root.showTranslationView = true
        }
    }
}
