pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import components.controls 1.0 as Controls

Dialog {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    required property var updateController
    required property var languageController
    required property var settingsModel

    // Local property for reactive changelog binding
    property var changelogData: []

    title: ""  // We use custom header

    Connections {
        target: root.updateController
        function onUpdateAvailableChanged() {
            if (root.updateController) {
                root.changelogData = root.updateController.changelogs()
            }
        }
    }

    onOpened: {
        if (root.updateController) {
            root.changelogData = root.updateController.changelogs()
        }
    }

    width: Math.min(parent.width * 0.9, 700)
    height: Math.min(parent.height * 0.85, 840)

    background: Rectangle {
        radius: Theme.panelRadius * 2
        color: Theme.surface
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    header: Rectangle {
        height: headerLayout.implicitHeight + Theme.spacingLg * 2
        color: "transparent"

        ColumnLayout {
            id: headerLayout
            anchors.fill: parent
            anchors.margins: Theme.spacingLg
            spacing: Theme.spacingSm

            RowLayout {
                Layout.fillWidth: true

                Label {
                    text: qsTr("Update Available")
                    font.pixelSize: 20
                    font.weight: Font.DemiBold
                    color: Theme.textPrimary
                    Layout.fillWidth: true
                }

                ToolButton {
                    id: closeButton
                    text: "\u2715"  // X symbol
                    font.pixelSize: 16
                    onClicked: root.close()

                    background: Rectangle {
                        radius: Theme.panelRadius
                        color: closeButton.hovered ? Theme.surfaceRaised : "transparent"
                    }
                }
            }

            RowLayout {
                spacing: Theme.spacingMd

                Label {
                    text: root.updateController ? root.updateController.currentVersion : ""
                    font.pixelSize: 14
                    color: Theme.textSecondary
                }

                Label {
                    text: "\u2192"  // Arrow
                    font.pixelSize: 14
                    color: Theme.textSecondary
                }

                Label {
                    text: root.updateController ? root.updateController.latestVersion : ""
                    font.pixelSize: 14
                    font.weight: Font.DemiBold
                    color: Theme.accent
                }
            }
        }

        Rectangle {
            anchors.bottom: parent.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            height: 1
            color: Theme.surfaceBorder
        }
    }

    contentItem: ColumnLayout {
        spacing: 0

        // Changelog section
        Label {
            text: qsTr("What's New")
            font.pixelSize: 14
            font.weight: Font.Medium
            color: Theme.textSecondary
            Layout.topMargin: Theme.spacingMd
            Layout.leftMargin: Theme.spacingLg
        }

        ScrollView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            Layout.margins: Theme.spacingMd
            clip: true

            ListView {
                id: changelogList
                spacing: Theme.spacingLg
                model: root.changelogData

                delegate: ColumnLayout {
                    id: versionDelegate
                    required property var modelData
                    width: ListView.view.width - Theme.spacingMd * 2

                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingMd

                        Label {
                            text: "v" + versionDelegate.modelData.version
                            font.pixelSize: 13
                            font.weight: Font.DemiBold
                            color: Theme.textPrimary
                        }

                        Label {
                            text: versionDelegate.modelData.date
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }

                        Item { Layout.fillWidth: true }
                    }

                    Repeater {
                        model: versionDelegate.modelData.changes

                        RowLayout {
                            id: changeDelegate
                            required property var modelData
                            Layout.fillWidth: true
                            spacing: Theme.spacingSm

                            Rectangle {
                                implicitWidth: 6
                                implicitHeight: 6
                                radius: 3
                                color: {
                                    const type = changeDelegate.modelData.type.toLowerCase()
                                    if (type === "feature") return Theme.accent
                                    if (type === "fix") return "#4caf50"
                                    return Theme.textSecondary
                                }
                                Layout.alignment: Qt.AlignTop
                                Layout.topMargin: 6
                            }

                            Label {
                                text: changeDelegate.modelData.text
                                font.pixelSize: 13
                                color: Theme.textPrimary
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }
                        }
                    }
                }
            }
        }
    }

    footer: Rectangle {
        height: footerLayout.implicitHeight + Theme.spacingLg * 2
        color: "transparent"

        Rectangle {
            anchors.top: parent.top
            anchors.left: parent.left
            anchors.right: parent.right
            height: 1
            color: Theme.surfaceBorder
        }

        RowLayout {
            id: footerLayout
            anchors.fill: parent
            anchors.margins: Theme.spacingLg
            spacing: Theme.spacingMd

            Controls.SecondaryButton {
                text: qsTr("Later")
                onClicked: root.close()
            }

            Controls.SecondaryButton {
                text: qsTr("Skip This Version")
                onClicked: {
                    if (root.settingsModel && root.updateController) {
                        root.settingsModel.set_ignored_update_version(root.updateController.latestVersion)
                    }
                    root.close()
                }
            }

            Item { Layout.fillWidth: true }

            Controls.PrimaryButton {
                text: qsTr("Open Downloads in Browser")
                onClicked: {
                    var lang = root.languageController ? root.languageController.currentLanguage : "en"
                    Qt.openUrlExternally(root.updateController.localizedDownloadUrl(lang))
                    root.close()
                }
            }
        }
    }
}
