pragma Singleton

import QtQuick 2.15
import SMC.Controllers 1.0

QtObject {
    id: theme

    // Bridge the context-provided controller into a typed property to appease qmllint
    // qmllint disable unqualified
    property ThemeController themeControllerRef: themeController
    // qmllint enable unqualified

    readonly property string mode: themeControllerRef ? themeControllerRef.mode : "light"
    readonly property bool isDark: mode === "dark"

    // Spacing scale (based on 4pt grid to align layouts across the app)
    readonly property real spacingXs: 4
    readonly property real spacingSm: 8
    readonly property real spacingMd: 12
    readonly property real spacingLg: 16
    readonly property real spacingXl: 24

    // Common layout tokens
    readonly property real pageMargin: spacingXl
    readonly property real panelRadius: 8

    // Base colors - single source of truth (matches AppRoot palette)
    readonly property color background: isDark ? "#2b2b2b" : "#f0f0f0"
    readonly property color surface: isDark ? "#3c3c3c" : "#ffffff"
    readonly property color surfaceRaised: isDark ? "#1e1e1e" : "#f5f5f5"
    readonly property color textPrimary: isDark ? "#ffffff" : "#000000"
    readonly property color accent: isDark ? "orange" : "#4c6ef5"
    readonly property color accentStrong: _mix(accent, _black, 0.2)


    // Derived colors using base colors above
    readonly property color surfaceBorder: isDark ? _withAlpha(textPrimary, 0.32) : _withAlpha(textPrimary, 0.12)
    readonly property color textSecondary: _withAlpha(textPrimary, 0.68)
    readonly property color textOnAccent: isDark ? _black : _withAlpha(_white, 1.0)
    readonly property color surfaceStrong: surfaceRaised

    // Status colors for alerts and highlights
    readonly property color statusDanger: isDark ? "#ff8787" : "#e03131"  // Use for destructive overlays like cut regions
    readonly property color statusSuccess: isDark ? "#4ade80" : "#22c55e"  // Use for success indicators

    readonly property color bottomBarBackground: isDark ? _mix(surface, _black, 0.28) : _mix(surface, _white, 0.16)
    readonly property color bottomBarBorder: _withAlpha(textPrimary, isDark ? 0.35 : 0.16)

    readonly property color toggleTrack: isDark ? _mix(surface, _black, 0.35) : _mix(surface, _white, 0.18)
    readonly property color toggleTrackBorder: _withAlpha(textPrimary, isDark ? 0.38 : 0.2)
    readonly property color toggleKnobActive: accent
    readonly property color toggleKnobActiveBorder: _mix(accent, _black, 0.35)
    readonly property color toggleKnobInactive: background
    readonly property color toggleKnobInactiveBorder: _withAlpha(textPrimary, isDark ? 0.32 : 0.18)

    // Derived colors for various surfaces
    readonly property color timelineBackground: isDark ? _mix(surface, _black, 0.32) : _mix(surface, _white, 0.12)
    readonly property color timelineRuler: isDark ? _mix(surface, _black, 0.22) : _mix(surface, _white, 0.18)
    readonly property color timelineTick: _withAlpha(textSecondary, isDark ? 0.85 : 0.78)
    readonly property color selectionHighlight: _withAlpha(accent, 0.22)
    readonly property color timelineSelectionBorder: accent
    readonly property color textSelection: _withAlpha(accent, isDark ? 0.35 : 0.9)
    readonly property color timelinePlayhead: accent
    readonly property color timelineCutOverlay: _withAlpha(statusDanger, 0.28)
    readonly property color timelineCutOverlayBorder: _withAlpha(statusDanger, isDark ? 0.9 : 0.75)
    // Loop uses distinct green color to differentiate from selection
    readonly property color loopColor: isDark ? "#4ade80" : "#22c55e"
    readonly property color timelineLoopFill: _withAlpha(loopColor, 0.18)
    readonly property color timelineLoopBorder: _withAlpha(loopColor, isDark ? 0.8 : 0.65)

    // Black/white detection pills
    readonly property color blackPillColor: isDark ? "#1a1a1a" : "#2d2d2d"
    readonly property color whitePillColor: isDark ? "#6a6a6a" : "#e0e0e0"
    readonly property color blackPillBorder: isDark ? "#333333" : "#1a1a1a"
    readonly property color whitePillBorder: isDark ? "#888888" : "#aaaaaa"

    // Detection area overlay
    readonly property color detectionOverlayDim: _withAlpha(_black, 0.5)
    readonly property color detectionAreaBorder: accent

    readonly property color _white: "#ffffff"
    readonly property color _black: "#000000"

    function iconUrl(name) {
        if (!name || name.length === 0)
            return ""
        var suffix = name.endsWith(".svg") ? name : name + ".svg"
        return Qt.resolvedUrl("../../icons/" + suffix)
    }

    function imageUrl(name) {
        if (!name || name.length === 0)
            return ""
        return Qt.resolvedUrl("../../images/" + name)
    }

    function _withAlpha(color, alpha) {
        return Qt.rgba(color.r, color.g, color.b, alpha)
    }

    function _mix(colorA, colorB, ratio) {
        var inv = 1 - ratio
        return Qt.rgba(colorA.r * inv + colorB.r * ratio,
                       colorA.g * inv + colorB.g * ratio,
                       colorA.b * inv + colorB.b * ratio,
                       colorA.a * inv + colorB.a * ratio)
    }

    function setMode(newMode) {
        if (newMode === mode)
            return
        if (themeControllerRef)
            themeControllerRef.setMode(newMode)
    }

    function toggle() {
        if (themeControllerRef)
            themeControllerRef.toggle()
    }

}
