pragma Singleton
import QtQuick

QtObject {
    // Normalize path slashes (no leading slash added)
    function normalizePath(path) {
        if (!path) return "";
        let normalized = String(path).replace(/\\/g, "/");
        normalized = normalized.replace(/\/+/g, "/");
        if (normalized.length > 1 && normalized.endsWith("/")) {
            normalized = normalized.slice(0, -1);
        }
        return normalized;
    }

    // Get directory portion of path
    function dirname(path) {
        const normalized = normalizePath(path);
        if (!normalized) return "";
        const trimmed = normalized.endsWith("/") && normalized.length > 1
            ? normalized.slice(0, -1) : normalized;
        const idx = trimmed.lastIndexOf("/");
        if (idx < 0) return "";
        // Handle Windows drive root (C:/)
        if (Qt.platform.os === "windows" && idx === 2 && normalized[1] === ":") {
            return normalized.slice(0, 3);
        }
        return trimmed.slice(0, idx) || "/";
    }

    // Convert filesystem path to file:// URL
    function pathToUrl(path) {
        if (!path) return "";
        const normalized = normalizePath(path);
        if (!normalized) return "";
        // Windows: C:/path -> file:///C:/path
        if (normalized.length > 1 && normalized[1] === ':') {
            return "file:///" + normalized;
        }
        // Unix: /path -> file:///path
        if (normalized.startsWith('/')) {
            return "file://" + normalized;
        }
        console.warn("PathUtils: relative path passed to pathToUrl:", path);
        return "file://" + normalized;
    }

    // Convert file:// URL to filesystem path
    function urlToPath(url) {
        if (!url) return "";
        let str = String(url);
        if (str.startsWith("file:///")) {
            str = decodeURIComponent(str.slice(7));
            // Windows: /C:/path -> C:/path
            if (str.length > 2 && str[0] === '/' && str[2] === ':') {
                str = str.slice(1);
            }
        } else if (str.startsWith("file://")) {
            str = decodeURIComponent(str.slice(7));
        }
        return normalizePath(str);
    }

    // Get folder URL from file path (for FileDialog.currentFolder)
    function pathToFolderUrl(path) {
        return pathToUrl(dirname(path));
    }

    // Format path for display with smart truncation
    // Adapts to path length: full path → .../parent/file → .../file
    function formatRecentPath(path, maxLength) {
        if (!maxLength) maxLength = 55;
        const normalized = normalizePath(path);
        if (!normalized) return "";

        // If full path fits, show it
        if (normalized.length <= maxLength) return normalized;

        const parts = normalized.split("/");
        const filename = parts[parts.length - 1] || normalized;

        // Try with parent directory
        if (parts.length > 2) {
            const parentDir = parts[parts.length - 2];
            const withParent = ".../" + parentDir + "/" + filename;
            if (withParent.length <= maxLength) return withParent;
        }

        // Fall back to just filename (for very long filenames)
        return ".../" + filename;
    }
}
